<?php

namespace Clonable\Translator\Model\Service;

use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\HTTP\Client\Curl;
use Clonable\Translator\Api\Service\ClonableGetSiteInfoApiInterface;
use Clonable\Translator\Model\ConfigManager;
use Magento\Store\Model\StoreManagerInterface;

class ClonableGetSiteInfoApi implements ClonableGetSiteInfoApiInterface
{
    /**
     * @var Curl
     */
    private Curl $curl;

    /**
     * @param ConfigManager $configManager
     */
    private ConfigManager $configManager;

    private StoreManagerInterface $storeManager;

    /**
     * @param Curl $curl
     * @param ConfigManager $configManager
     */
    public function __construct
    (
        Curl $curl,
        ConfigManager $configManager,
        StoreManagerInterface $storeManager
    ){
        $this->curl = $curl;
        $this->configManager = $configManager;
        $this->storeManager = $storeManager;
    }

    /**
     * @param $apiKey
     * @param null $storeId
     * @return mixed
     * @throws LocalizedException
     */
    public function getSiteInfo($apiKey = null, $storeId = null)
    {
        $apiKey = $apiKey ?? $this->configManager->decryptApiKey($this->configManager->getApiKey($storeId));
        $maxAttempts = 5;
        $initialDelay = 1;
        $url = ConfigManager::API_URL . '/sites/' . $this->getDomainWithoutProtocol();
        $headers = [
            'Authorization' => 'Bearer ' . $apiKey,
            'Content-Type' => 'application/json',
        ];

        for ($attempt = 0; $attempt < $maxAttempts; $attempt++) {
            try {
                $this->curl->setHeaders($headers);
                $this->curl->get($url);

                if ($this->curl->getStatus() === 200) {
                    return json_decode($this->curl->getBody());
                }
            } catch (\Exception $e) {
                if ($attempt >= $maxAttempts - 1) {
                    throw new LocalizedException(__('Error fetching site info: ' . $e->getMessage()));
                }

                sleep($initialDelay * pow(2, $attempt));
            }
        }

        throw new LocalizedException(__('Failed to get Site Info after multiple attempts.'));
    }

    /**
     * @param $siteInfo
     */
    public function getSiteId($siteInfo)
    {
        return $siteInfo->site->id;
    }

    /**
     * @param $siteInfo
     */
    public function getOriginLanguage($siteInfo)
    {
        return explode('_',$siteInfo->site->lang_code)[0] ?? '';
    }

    /**
     * @param $siteInfo
     * @param $storeCode
     */
    public function getTargetLanguage($siteInfo, $storeCode)
    {
        foreach ($siteInfo->site->clones as $clone) {
            if (trim($clone->subfolder_clone, '/') == $storeCode) {
                return explode('_',$clone->lang_code)[0] ?? '';
            }
        }
    }

    /**
     * @param $siteInfo
     * @param $storeCode
     */
    public function getCloneId($siteInfo, $storeCode)
    {
        foreach ($siteInfo->site->clones as $clone) {
            if (trim($clone->subfolder_clone, '/') == $storeCode) {
                return $clone->id ?? '';
            }
        }
    }

    public function getDomainWithoutProtocol()
    {

        $baseUrl = $this->storeManager->getStore()->getBaseUrl();

        $parsedUrl = parse_url($baseUrl);

        return $parsedUrl['host'];
    }

    public function getSubfolderClone($siteInfo, $storeCode): ?string
    {
        foreach ($siteInfo->site->clones as $clone) {
            if (trim($clone->subfolder_clone, '/') == $storeCode) {
                return $clone->subfolder_clone ?? null;
            }
        }
        return null;
    }
}
