<?php

namespace Clonable\Translator\Model;

use Magento\Catalog\Model\Product;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\Encryption\EncryptorInterface;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;

class ConfigManager
{
    public const IS_ENABLED = 'translator/general/is_enabled';
    public const API_KEY = 'translator/general/api_key';
    public const AMQP_CONNECTION = 'translator/general/amqp';
    public const TRANSLATE_URL_KEY = 'translator/general/enable_translate_url_key';
    public const OVERRIDE_TRANSLATION_DICTIONARY_KEY = 'translator/general/override_dictionary_key';
    public const USE_NAME_BASED_URL_KEY_TRANSLATIONS = 'translator/general/use_name_based_url_key';
    public const TOPIC_PRODUCT_TRANSLATOR_DB = 'clonable.translator.product.queue.db';
    public const TOPIC_CATEGORY_TRANSLATOR_DB = 'clonable.translator.category.queue.db';
    public const TOPIC_CMS_PAGE_TRANSLATOR_DB = 'clonable.translator.cms.queue.db';
    public const TOPIC_PRODUCT_TRANSLATOR_AMQP = 'clonable.translator.product.queue.amqp';
    public const TOPIC_CATEGORY_TRANSLATOR_AMQP = 'clonable.translator.category.queue.amqp';
    public const TOPIC_CMS_PAGE_TRANSLATOR_AMQP = 'clonable.translator.cms.queue.amqp';
    public const API_URL = 'https://api.clonable.net';
    public const API_URL_TRANSLATE = 'https://ut.api.clonable.net/s/translation/create';
    public const API_URL_TEXT = 'https://translations.api.clonable.net/translate';
    public const ENABLE_AUTO_TRANSLATION = 'translator/general/enable_auto_translation';
    public const CLONE_ID = 'clonable/general/clone_id';
    public const SITE_ID = 'clonable/general/site_id';
    public const ORIGINAL_LANGUAGE = 'clonable/general/original_language';
    public const TARGET_LANGUAGE = 'clonable/general/target_language';
    public const SUBFOLDER_CLONE = 'clonable/general/subfolder_clone';

    /**
     * @var EncryptorInterface
     */
    private EncryptorInterface $encryptor;

    /**
     * @var ScopeConfigInterface
     */
    private ScopeConfigInterface $scopeConfig;

    private StoreManagerInterface $storeManager;

    private WriterInterface $configWriter;

    /**
     * @param ScopeConfigInterface $scopeConfig
     */
    public function __construct(
        ScopeConfigInterface $scopeConfig,
        EncryptorInterface $encryptor,
        StoreManagerInterface $storeManager,
        WriterInterface $configWriter
    ){
        $this->scopeConfig = $scopeConfig;
        $this->encryptor = $encryptor;
        $this->storeManager = $storeManager;
        $this->configWriter = $configWriter;
    }

    /**
     * @param null $store
     * @return bool
     */
    public function isEnabled($store = null): bool
    {
        return $this->scopeConfig->isSetFlag(self::IS_ENABLED, ScopeInterface::SCOPE_STORES, $store);
    }

    /**
     * @param null $store
     * @return string|null
     */
    public function getApiKey($store = null): ?string
    {
        return $this->scopeConfig->getValue(self::API_KEY, ScopeInterface::SCOPE_STORE, $store);
    }

    /**
     * @param null $store
     * @return bool
     */
    public function useRabbitMq($store = null): bool
    {
        return $this->scopeConfig->isSetFlag(self::AMQP_CONNECTION, ScopeInterface::SCOPE_STORES, $store);
    }

    public function getTopicQueueForProduct(): string
    {
        return $this->useRabbitMq() ? self::TOPIC_PRODUCT_TRANSLATOR_AMQP : self::TOPIC_PRODUCT_TRANSLATOR_DB;
    }

    public function getTopicQueueForCategory(): string
    {
        return $this->useRabbitMq() ? self::TOPIC_CATEGORY_TRANSLATOR_AMQP : self::TOPIC_CATEGORY_TRANSLATOR_DB;
    }

    public function getTopicQueueForCmsPage(): string
    {
        return $this->useRabbitMq() ? self::TOPIC_CMS_PAGE_TRANSLATOR_AMQP : self::TOPIC_CMS_PAGE_TRANSLATOR_DB;
    }

    /**
     *
     * @param Product $product
     * @return bool
     */
    public function isExcludedFromTranslation(Product $product): bool
    {
        return $product->getExcludeClonableAutotranslation() ?? false;
    }

    /**
     * @param $apiKey
     * @return string
     */
    public function decryptApiKey($apiKey): string
    {
        return $this->encryptor->decrypt($apiKey);
    }

    public function encryptApiKey($apiKey): string
    {
        return $this->encryptor->encrypt($apiKey);
    }
    /**
     * @param null $store
     * @return bool
     */
    public function isEnabledTranslateUrlKey($store = null): bool
    {
        return $this->scopeConfig->isSetFlag(self::TRANSLATE_URL_KEY, ScopeInterface::SCOPE_STORES, $store);
    }

    /**
     * @param null $store
     * @return bool
     */
    public function isEnabledOverrideTranslationDictionary($store = null): bool
    {
        return $this->scopeConfig->isSetFlag(self::OVERRIDE_TRANSLATION_DICTIONARY_KEY, ScopeInterface::SCOPE_STORE, $store);
    }

    /**
     * @param $store
     * @return bool
     */
    public function useNameBasedUrlKeyTranslation($store = null): bool
    {
        return $this->scopeConfig->isSetFlag(self::USE_NAME_BASED_URL_KEY_TRANSLATIONS, ScopeInterface::SCOPE_STORE, $store);
    }

    /**
     * @return array
     */
    public function getStoresForTranslate(): array
    {
        $storeIds = [];

        foreach ($this->storeManager->getStores() as $store) {
            if ($this->getApiKey($store->getId()) && $this->isEnabled($store->getId())) {
                $storeIds[] = $store->getId();
            }
        }

        return $storeIds;
    }

    /**
     * @param null $store
     * @return bool
     */
    public function isEnabledAutoTranslation($store = null): bool
    {
        return $this->scopeConfig->isSetFlag(self::ENABLE_AUTO_TRANSLATION, ScopeInterface::SCOPE_STORES, $store);
    }

    public function getCloneId($store = null): ?string
    {
        return $this->scopeConfig->getValue(self::CLONE_ID, ScopeInterface::SCOPE_STORES, $store);
    }

    public function getSiteId($store = null): ?string
    {
        return $this->scopeConfig->getValue(self::SITE_ID, ScopeInterface::SCOPE_STORES, $store);
    }

    public function getOriginalLanguage($store = null): ?string
    {
        return $this->scopeConfig->getValue(self::ORIGINAL_LANGUAGE, ScopeInterface::SCOPE_STORES, $store);
    }

    public function getTargetLanguage($store = null): ?string
    {
        return $this->scopeConfig->getValue(self::TARGET_LANGUAGE, ScopeInterface::SCOPE_STORES, $store);
    }

    public function getSubfolderClone($store = null): ?string
    {
        return $this->scopeConfig->getValue(self::SUBFOLDER_CLONE, ScopeInterface::SCOPE_STORES, $store);
    }

    public function setCloneId($value, $store = null): void
    {
        $this->configWriter->save(self::CLONE_ID, $value, ScopeInterface::SCOPE_STORES, $store);
    }

    public function setSiteId($value, $store = null): void
    {
        $this->configWriter->save(self::SITE_ID, $value, ScopeInterface::SCOPE_STORES, $store);
    }

    public function setOriginalLanguage($value, $store = null): void
    {
        $this->configWriter->save(self::ORIGINAL_LANGUAGE, $value, ScopeInterface::SCOPE_STORES, $store);
    }

    public function setTargetLanguage($value, $store = null): void
    {
        $this->configWriter->save(self::TARGET_LANGUAGE, $value, ScopeInterface::SCOPE_STORES, $store);
    }

    public function setSubfolderClone($value, $store = null): void
    {
        $this->configWriter->save(self::SUBFOLDER_CLONE, $value, ScopeInterface::SCOPE_STORES, $store);
    }

    public function getLocale($store_id): ?string
    {
        $locale = $this->scopeConfig->getValue(
            'general/locale/code',
            ScopeInterface::SCOPE_STORE,
            $store_id
        );

        if (!$locale) {
            return null;
        }

        return $locale;
    }

}
