<?php

namespace Clonable\Translator\Model\Service;

use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\HTTP\Client\Curl;
use Clonable\Translator\Api\Service\ClonableGetSiteInfoApiInterface;
use Clonable\Translator\Model\ConfigManager;
use Magento\Store\Model\StoreManagerInterface;

class ClonableGetSiteInfoApi implements ClonableGetSiteInfoApiInterface {
    /**
     * @var Curl
     */
    private Curl $curl;

    /**
     * @param ConfigManager $configManager
     */
    private ConfigManager $configManager;

    private StoreManagerInterface $storeManager;

    /**
     * @param Curl $curl
     * @param ConfigManager $configManager
     */
    public function __construct
    (
        Curl                  $curl,
        ConfigManager         $configManager,
        StoreManagerInterface $storeManager
    ) {
        $this->curl = $curl;
        $this->configManager = $configManager;
        $this->storeManager = $storeManager;
    }

    /**
     * @param $apiKey
     * @param null $storeId
     * @return mixed
     * @throws LocalizedException
     */
    public function getSiteInfo($apiKey = null, $storeId = null) {
        $apiKey = $apiKey ?? $this->configManager->decryptApiKey($this->configManager->getApiKey($storeId));
        $maxAttempts = 5;
        $initialDelay = 1;
        $url = ConfigManager::API_URL . '/sites/' . $this->getDomainWithoutProtocol();
        $headers = [
            'Authorization' => 'Bearer ' . $apiKey,
            'Content-Type' => 'application/json',
        ];

        for ($attempt = 0; $attempt < $maxAttempts; $attempt++) {
            try {
                $this->curl->setHeaders($headers);
                $this->curl->get($url);

                if ($this->curl->getStatus() === 200) {
                    return json_decode($this->curl->getBody());
                }
            } catch (\Exception $e) {
                if ($attempt >= $maxAttempts - 1) {
                    throw new LocalizedException(__('Error fetching site info: ' . $e->getMessage()));
                }

                sleep($initialDelay * pow(2, $attempt));
            }
        }

        throw new LocalizedException(__('Failed to get Site Info after multiple attempts.'));
    }

    /**
     * @param $siteInfo
     */
    public function getSiteId($siteInfo) {
        return $siteInfo->site->id;
    }

    /**
     * @param $siteInfo
     */
    public function getOriginLanguage($siteInfo) {
        return explode('_', $siteInfo->site->lang_code)[0] ?? '';
    }

    /**
     * @param mixed $siteInfo
     * @param string $storeCode
     * @param string $baseUrl
     * @return mixed|string
     */
    public function getTargetLanguage($siteInfo, string $storeCode, string $baseUrl) {
        if ($clone = $this->getMatchingClone($siteInfo, $storeCode, $baseUrl)) {
            return explode('_', $clone->lang_code)[0] ?? '';
        } else {
            return '';
        }
    }

    /**
     * @param $siteInfo
     * @param $storeCode
     * @param $baseUrl
     * @return string
     */
    public function getCloneId($siteInfo, $storeCode, $baseUrl) {
        if ($clone = $this->getMatchingClone($siteInfo, $storeCode, $baseUrl)) {
            return $clone->id ?? '';
        } else {
            return '';
        }
    }

    public function getDomainWithoutProtocol() {

        $baseUrl = $this->storeManager->getStore()->getBaseUrl();

        $parsedUrl = parse_url($baseUrl);

        return $parsedUrl['host'];
    }

    /**
     * @param mixed $siteInfo the information from the api
     * @param string $storeCode the storeCode
     * @param string $baseUrl the base url of the StoreView
     * @return string|null
     */
    public function getSubfolderClone($siteInfo, string $storeCode, string $baseUrl): ?string {
        if ($clone = $this->getMatchingClone($siteInfo, $storeCode, $baseUrl)) {
            return $clone->subfolder_clone;
        } else {
            return null;
        }
    }

    /**
     * Attempts to find a matching clone in the api siteInfo response. If no matching clone is found, null is returned.
     *
     *
     * @param $siteInfo
     * @param string $storeCode Store code of store
     * @param string $baseUrl Full base url of store
     * @return object|null
     */
    private function getMatchingClone($siteInfo, $storeCode, $baseUrl) {
        $parsed_base_url = parse_url($baseUrl);
        if (substr($parsed_base_url['host'], 0, strlen('www.')) === 'www.') {
            $store_domain_no_www = substr($parsed_base_url['host'], strlen('www.'));
        } else {
            $store_domain_no_www = $parsed_base_url['host'];
        }

        // First look for matching subfolder clone
        foreach ($siteInfo->site->clones as $clone) {
            if ($clone->is_subfolder_clone) {
                // if the clone is a subfolder clone, then we have to be careful on how to match the domain.
                $clone_domain_and_subfolder = $clone->domain . ($clone->subfolder_clone ?? '');
                if ($parsed_base_url['path'] === '/') {
                    // if the base url does not have a path, we'll inject the store code in the url.
                    if (($store_domain_no_www . "/$storeCode/") === $clone_domain_and_subfolder) {
                        return $clone;
                    }
                } else if ($clone_domain_and_subfolder === ($store_domain_no_www . $parsed_base_url['path'])) {
                    return $clone;
                }
            }
        }

        // If no matching subfolder clone is found, search on domain match only
        foreach ($siteInfo->site->clones as $clone) {
            if (!$clone->is_subfolder_clone) {
                if ($clone->domain === $store_domain_no_www) {
                    return $clone;
                }
            }
        }

        return null;
    }
}
