<?php

namespace Clonable\Translator\Model;

use Magento\Framework\MessageQueue\QueueInterface;
use Magento\Framework\App\CacheInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Store\Model\Store;
use Magento\UrlRewrite\Model\UrlPersistInterface;
use Magento\UrlRewrite\Model\UrlFinderInterface;
use Clonable\Translator\Api\Data\TranslatorInterface;
use Clonable\Translator\Api\Service\ClonableTranslatorApiInterface;


abstract class AbstractTranslator implements TranslatorInterface
{
    /**
     * @var UrlPersistInterface
     */
    protected UrlPersistInterface $urlPersist;

    /**
     * @var UrlFinderInterface
     */
    protected UrlFinderInterface $urlFinder;

    /**
     * @var CacheInterface
     */
    protected CacheInterface $cache;

    /**
     * @var Json
     */
    protected Json $json;

    /**
     * @var StoreManagerInterface
     */
    protected StoreManagerInterface $storeManager;

    /**
     * @var ConfigManager
     */
    protected ConfigManager $configManager;

    /**
     * @var ReportLogsFactory
     */
    protected ReportLogsFactory $reportLogsFactory;

    /**
     * @var ClonableTranslatorApiInterface
     */
    private ClonableTranslatorApiInterface $clonableTranslatorAPIInterface;

    /**
     * @param UrlPersistInterface $urlPersist
     * @param UrlFinderInterface $urlFinder
     * @param CacheInterface $cache
     * @param Json $json
     * @param StoreManagerInterface $storeManager
     * @param ConfigManager $configManager
     * @param ClonableTranslatorApiInterface $clonableTranslatorAPIInterface
     */
    public function __construct
    (
        UrlPersistInterface $urlPersist,
        UrlFinderInterface $urlFinder,
        CacheInterface $cache,
        Json $json,
        StoreManagerInterface $storeManager,
        ConfigManager $configManager,
        ClonableTranslatorApiInterface $clonableTranslatorAPIInterface,
        ReportLogsFactory $reportLogsFactory
    ){
        $this->urlPersist = $urlPersist;
        $this->urlFinder = $urlFinder;
        $this->cache = $cache;
        $this->json = $json;
        $this->storeManager = $storeManager;
        $this->configManager = $configManager;
        $this->clonableTranslatorAPIInterface = $clonableTranslatorAPIInterface;
        $this->reportLogsFactory = $reportLogsFactory;
    }

    /**
     * @param string $text
     * @param Store $store
     * @return string
     */
    public function translateText(string $text, Store $store): string
    {
        return ucfirst($this->clonableTranslatorAPIInterface->translateText($text, $store));
    }

    /**
     * @param string $urlKey
     * @param Store $store
     * @return string
     */
    public function translateUrlKey(string $urlKey, Store $store): string
    {
        return $this->clonableTranslatorAPIInterface->translateUrlKey($urlKey, $store);
    }

    /**
     * @param QueueInterface $queue
     * @return void
     **/
    public abstract function processMessage(QueueInterface $queue): void;

    /**
     * @return array
     */
    protected function getStoresForTranslate(): array
    {
        $storeIds = [];

        foreach ($this->storeManager->getStores() as $store) {
            if ($this->configManager->getApiKey($store->getId()) && $this->configManager->isEnabled($store->getId())) {
                $storeIds[] = $store->getId();
            }
        }

        return $storeIds;
    }

    /**
     *
     * @param string $message
     * @param string $storeId
     * @return ReportLogs
     */
    protected function createReportLog(string $message, string $storeId = '0'): ReportLogs {
        $reportLog = $this->reportLogsFactory->create();
        $reportLog->setMessage($message);
        $reportLog->setStoreId($storeId);
        return $reportLog;
    }
}
