<?php

namespace Clonable\Translator\Model\Category;

use Clonable\Translator\Api\Data\Category\TranslatorCategoryInterface;
use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Api\Service\ClonableTranslatorApiInterface;
use Clonable\Translator\Model\AbstractTranslator;
use Clonable\Translator\Model\ConfigManager;
use Clonable\Translator\Model\Logger\Logger;
use Clonable\Translator\Model\Category\Condition\ConditionChain;
use Clonable\Translator\Model\ReportLogsFactory;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\CatalogUrlRewrite\Model\CategoryUrlRewriteGenerator;
use Magento\Framework\App\CacheInterface;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\MessageQueue\QueueInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use Magento\UrlRewrite\Model\Exception\UrlAlreadyExistsException;
use Magento\UrlRewrite\Model\UrlFinderInterface;
use Magento\UrlRewrite\Model\UrlPersistInterface;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite as UrlDataRewrite;
use Magento\UrlRewrite\Service\V1\Data\UrlRewriteFactory as UrlRewriteDataFactory;
use Magento\UrlRewrite\Model\ResourceModel\UrlRewriteCollectionFactory;
use Throwable;

class TranslatorCategory extends AbstractTranslator implements TranslatorCategoryInterface
{
    private CategoryRepositoryInterface $categoryRepository;
    private CategoryUrlRewriteGenerator $categoryUrlRewriteGenerator;
    private ReportLogsRepositoryInterface $reportLogsRepository;
    private UrlRewriteDataFactory $urlRewriteDataFactory;
    private UrlRewriteCollectionFactory $urlRewriteCollectionFactory;
    private ConditionChain $conditionChain;
    private Logger $logger;

    public function __construct(
        UrlPersistInterface            $urlPersist,
        UrlFinderInterface             $urlFinder,
        CacheInterface                 $cache,
        CategoryRepositoryInterface    $categoryRepository,
        CategoryUrlRewriteGenerator    $categoryUrlRewriteGenerator,
        Json                           $json,
        Logger                         $logger,
        StoreManagerInterface          $storeManager,
        ConfigManager                  $configManager,
        ClonableTranslatorApiInterface $clonableTranslatorAPIInterface,
        ReportLogsRepositoryInterface  $reportLogsRepository,
        ConditionChain                 $conditionChain,
        UrlRewriteCollectionFactory    $urlRewriteCollectionFactory,
        ReportLogsFactory              $reportLogsFactory,
        UrlRewriteDataFactory          $urlRewriteFactory
    )
    {
        parent::__construct($urlPersist, $urlFinder, $cache, $json, $storeManager, $configManager, $clonableTranslatorAPIInterface, $reportLogsFactory);
        $this->categoryRepository = $categoryRepository;
        $this->categoryUrlRewriteGenerator = $categoryUrlRewriteGenerator;
        $this->logger = $logger;
        $this->reportLogsRepository = $reportLogsRepository;
        $this->conditionChain = $conditionChain;
        $this->urlRewriteDataFactory = $urlRewriteFactory;
        $this->urlRewriteCollectionFactory = $urlRewriteCollectionFactory;
    }

    private function print(string $message, string $type = 'error')
    {
        if ($type == 'error') {
            $this->logger->error($message);
            $this->reportLogsRepository->save($this->createReportLog($message));
        } else if ($type === 'warning') {
            $this->logger->warning($message);
        } else if ($type === 'info') {
            $this->logger->info($message);
        } else if ($type === 'debug') {
            $this->logger->debug($message);
        }
    }

    public function processMessage(QueueInterface $queue): void
    {
        if ($message = $queue->dequeue()) {
            try {
                $data = $this->json->unserialize($this->json->unserialize($message->getBody()));
                if (!is_array($data) || !array_key_exists('categoryId', $data) || !array_key_exists('storeId', $data) || !array_key_exists('force', $data)) {
                    $this->print("Invalid data: " . print_r($data, true));
                    $queue->reject($message, false, "Invalid data");
                    return;
                }

                // Prepare and parse serialized data
                $category_id = intval($data['categoryId']);
                $storeId = intval($data['storeId']);
                $force_translation = boolval($data['force']);

                if ($category_id == 0 || $storeId == Store::DEFAULT_STORE_ID) {
                    $this->print('No store IDs available for translation category ID ' . $category_id, 'warning');
                    $queue->reject($message, false, "Invalid data");
                    return;
                }

                try {
                    // Retrieve the store, category and base category for translations
                    $store = $this->storeManager->getStore($storeId);
                    $this->storeManager->setCurrentStore($store);
                    $category = $this->categoryRepository->get($category_id, $storeId);
                } catch (NoSuchEntityException $e) {
                    $this->print("Failed to load category $category_id for store $storeId");
                    $queue->reject($message, false, "Non-existent category $category_id for store $storeId");
                    return;
                }

                if ($this->conditionChain->shouldProcess($category, $force_translation)) {
                    $this->translateCategoryUrlKey($category, $store);
                }

                $queue->acknowledge($message);
            } catch (Throwable $e) {
                $this->print('Error updating category with ID ' . ($category_id ?? 'N/A') . ' in store ' . ($storeId ?? 'N/A') . ': ' . $e->getMessage());
                $queue->reject($message, true, $e->getMessage());
                sleep(1); // Sleep to prevent retry storm
            }
        } else {
            usleep(100_000);
        }
    }

    /**
     * The logic of translating the category and create the underlying rewrites.
     * @param CategoryInterface $category the category to translate the url key for
     * @param Store $store the store of the category
     * @return void
     */
    private function translateCategoryUrlKey(CategoryInterface $category, Store $store)
    {
        try {
            $original_category = $this->categoryRepository->get($category->getId(), Store::DEFAULT_STORE_ID);
        } catch (NoSuchEntityException $e) {
            $this->print("Could not find the root category for {$category->getName()}: " . $e->getMessage());
            return;
        }

        // Always translate the url key based on the original, this solves problems with re-translation.
        $translated_url_key = $this->translateUrlKey($original_category->getUrlKey(), $store);

        try {
            // Set the category data
            $category = $this->categoryRepository->get($category->getId(), $store->getId());
            $category->setStoreId($store->getId());
            $category->setExcludeClonableAutoTranslation(true);
            $category->setUrlKey($translated_url_key);
            $this->categoryRepository->save($category);
        } catch (CouldNotSaveException|NoSuchEntityException $e) {
            $this->print("{$category->getName()}({$category->getId()}) -> Could not save category after setting new url key data:  " . $e->getMessage());
            return;
        }

        try {
            // Retrieve the category again to solve (possible) local cache problems
            $category = $this->categoryRepository->get($category->getId(), $store->getId());
            $category = $category->setUrlPath($this->buildTranslatedPath($category, $store));
        } catch (NoSuchEntityException $e) {
            $this->print("{$category->getName()}({$category->getId()}) -> Could not retrieve category the after saving:  " . $e->getMessage());
            return;
        }

        try {
            // Generate the rewrites and persist them
            $rewrites = $this->categoryUrlRewriteGenerator->generate($category);

            // make sure to redirect the original to the translated
            $rewrite = $this->getRedirect($category, $store, $rewrites);
            if ($rewrite) {
                $rewrites[] = $rewrite;
            }

            $this->urlPersist->replace($rewrites);
        } catch (UrlAlreadyExistsException $e) {
            $this->print("{$category->getName()}({$category->getId()}) -> Duplicate url detected for category");
        } catch (\Throwable $e) {
            $this->print("{$category->getName()}({$category->getId()}) -> Unknown error occurred while persisting urls:  " . $e->getMessage());
        }
    }

    /**
     * @param CategoryInterface $category
     * @param Store $store
     * @param UrlDataRewrite[] $rewrites
     * @return UrlDataRewrite|null
     */
    private function getRedirect(CategoryInterface $category, Store $store, array $rewrites): ?UrlDataRewrite
    {
        try {
            $original_path = $this->buildOriginalPath($category, $store);
            $redirect_data_object = $this->getRedirectDataObject($category, $store, $original_path);
        } catch (NoSuchEntityException $e) {
            // don't bother creating one if it's going to fail.
            // redirects are technically not required for the correct translations of the urls keys
            return null;
        }

        // Make sure to check the auto generated rewrites for conflicting paths.
        foreach ($rewrites as $rewrite) {
            if ($rewrite->getRequestPath() === $original_path) {
                return null;
            }
        }

        // Make a collection query to check existing rewrites for this category
        $collection = $this->urlRewriteCollectionFactory->create()
            ->addFieldToFilter('entity_type', 'category')
            ->addFieldToFilter('entity_id', $category->getId())
            ->addFieldToFilter('store_id', $store->getId())
            ->addFieldToFilter('request_path', $original_path)
            ->addFieldToFilter('redirect_type', 301);

        $existing_rewrite = $collection->getFirstItem();
        if ($existing_rewrite->getId()) {
            return null; // if a rewrite for this exact category already exists, then ignore the creation.
        }
        return $this->urlRewriteDataFactory->create(['data' => $redirect_data_object]);
    }

    /**
     * Get the redirect rewrite as a data object.
     * This function relies on the category url_path to be translated.
     * If the path is not translated, and untranslated redirect will be created.
     */
    private function getRedirectDataObject(CategoryInterface $category, Store $store, string $original_path): array
    {
        return [
            UrlDataRewrite::ENTITY_TYPE => 'category',
            UrlDataRewrite::ENTITY_ID => $category->getId(),
            UrlDataRewrite::STORE_ID => $store->getId(),
            UrlDataRewrite::REQUEST_PATH => $original_path,
            UrlDataRewrite::TARGET_PATH => $category->getData('url_path'),
            UrlDataRewrite::REDIRECT_TYPE => 301,
            UrlDataRewrite::IS_AUTOGENERATED => 0,
        ];
    }

    /**
     * Build the new url path for the category.
     * This will automatically translate all the path segments for the category.
     * This directly solves issues with partly translated path segment,
     * because each segment is checked against the url translations
     *
     * @param CategoryInterface $category the category to create the translated path for
     * @param Store $store the store of the category
     * @return string  The slash-joined, translated segments for this category’s full path
     * @throws NoSuchEntityException when the root category of the store cannot be retrieved
     */
    private function buildTranslatedPath(CategoryInterface $category, Store $store): string
    {
        $ids = $this->getPathIds($category, $store);

        $translated_segments = [];
        foreach ($ids as $id) {
            // Translate url key if it was not translated yet
            $untranslated_ancestor = $this->categoryRepository->get($id, Store::DEFAULT_STORE_ID);
            $translated = $this->translateUrlKey($untranslated_ancestor->getUrlKey(), $store);

            $translated_segments[] = $translated;
        }

        return implode('/', $translated_segments);
    }

    /**
     * Build the original path like the translated path.
     * But retrieve the url key instead of making an API call.
     * @throws NoSuchEntityException could not find root category
     */
    private function buildOriginalPath(CategoryInterface $category, Store $store): string
    {
        $parent = $this->categoryRepository->get($category->getId(), Store::DEFAULT_STORE_ID);
        $ids = $this->getPathIds($parent, $store);
        $segments = [];
        foreach ($ids as $id) {
            // Retrieve the url key of the untranslated category
            $ancestor = $this->categoryRepository->get($id, Store::DEFAULT_STORE_ID);
            $segments[] = $ancestor->getUrlKey();
        }

        return implode('/', $segments);
    }

    /**
     * Helpers method for getting the correct ids based on the category path.
     * Automatically filters out the default and root categories.
     * @throws NoSuchEntityException the root category could not be found.
     */
    private function getPathIds(CategoryInterface $category, Store $store): array
    {
        $root_category_id = $store->getRootCategoryId();

        // map ids to int and filter out the unwanted ids
        $parts = explode('/', $category->getPath());
        $ids = array_map(fn($id) => intval($id), $parts);
        return array_filter($ids, function ($id) use ($root_category_id) {
            return !($id === 1 || $id == $root_category_id); // make sure not to translate the root
        });
    }
}
