<?php

namespace Clonable\Translator\Model\Category;

use Clonable\Translator\Api\Data\Category\TranslatorCategoryInterface;
use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Api\Service\ClonableTranslatorApiInterface;
use Clonable\Translator\Model\AbstractTranslator;
use Clonable\Translator\Model\ConfigManager;
use Clonable\Translator\Model\Logger\Logger;
use Clonable\Translator\Model\Category\Condition\ConditionChain;
use Clonable\Translator\Model\ReportLogsFactory;
use Exception;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\CatalogUrlRewrite\Model\CategoryUrlRewriteGenerator;
use Magento\Framework\App\CacheInterface;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\MessageQueue\QueueInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\StoreManagerInterface;
use Magento\UrlRewrite\Model\Exception\UrlAlreadyExistsException;
use Magento\UrlRewrite\Model\UrlFinderInterface;
use Magento\UrlRewrite\Model\UrlPersistInterface;
use Throwable;

class TranslatorCategory extends AbstractTranslator implements TranslatorCategoryInterface
{
    private CategoryRepositoryInterface $categoryRepository;
    private CategoryUrlRewriteGenerator $categoryUrlRewriteGenerator;
    private ReportLogsRepositoryInterface $reportLogsRepository;
    private ConditionChain $conditionChain;
    private Logger $logger;


    public function __construct(
        UrlPersistInterface            $urlPersist,
        UrlFinderInterface             $urlFinder,
        CacheInterface                 $cache,
        CategoryRepositoryInterface    $categoryRepository,
        CategoryUrlRewriteGenerator    $categoryUrlRewriteGenerator,
        Json                           $json,
        Logger                         $logger,
        StoreManagerInterface          $storeManager,
        ConfigManager                  $configManager,
        ClonableTranslatorApiInterface $clonableTranslatorAPIInterface,
        ReportLogsRepositoryInterface  $reportLogsRepository,
        ReportLogsFactory              $reportLogsFactory,
        ConditionChain       $conditionChain
    ) {
        parent::__construct($urlPersist, $urlFinder, $cache, $json, $storeManager, $configManager, $clonableTranslatorAPIInterface, $reportLogsFactory);
        $this->categoryRepository = $categoryRepository;
        $this->categoryUrlRewriteGenerator = $categoryUrlRewriteGenerator;
        $this->logger = $logger;
        $this->reportLogsRepository = $reportLogsRepository;
        $this->conditionChain = $conditionChain;
    }

    public function processMessage(QueueInterface $queue): void
    {
        if ($message = $queue->dequeue()) {
            try {
                $data = $this->json->unserialize($this->json->unserialize($message->getBody()));
                if (!is_array($data) || !array_key_exists('categoryId', $data) || !array_key_exists('storeId', $data) || !array_key_exists('force', $data)) {
                    $this->logger->error("Invalid data: " . print_r($data, true));
                    $queue->reject($message, false, "Invalid data");
                    return;
                }

                $categoryId = intval($data['categoryId']);
                $storeId = intval($data['storeId']);
                $force = $data['force'];

                if ($categoryId == 0 || $storeId == 0) {
                    $this->logger->warning('No store IDs available for translation category ID ' . $categoryId);
                    $this->reportLogsRepository->save($this->createReportLog('No store IDs available for translation category SKU ' . $categoryId));
                    $queue->reject($message, false, "Invalid data");
                    return;
                }

                $this->storeManager->setCurrentStore($storeId);
                // Retrieve the base category for translations
                try {
                    $category = $this->categoryRepository->get($categoryId, $storeId);
                } catch (NoSuchEntityException $e) {
                    $this->logger->error("Failed to load (base) category $categoryId for store $storeId");
                    $this->reportLogsRepository->save($this->createReportLog("Failed to load (base) category $categoryId for store $storeId"));
                    $queue->reject($message, false, "Non-existent category $categoryId for store $storeId");
                    return;
                }

                if ($this->conditionChain->shouldProcess($category, $force)) {
                    $this->logger->info("Translating category $categoryId ({$category->getName()}) for store {$category->getStore()->getId()}");

                    $startScriptTime = microtime(true);
                    $category = $this->updateCategoryAttributes($category, $storeId);
                    $this->updateUrlRewrite($category, $storeId);

                    $totalTime = microtime(true) - $startScriptTime;
                    $this->logger->info("Translated category {$categoryId} in store {$storeId}: {$totalTime} seconds");
                } else {
                    $this->logger->info("Not translating category $categoryId ({$category->getName()}) for store {$category->getStore()->getId()} because of failed conditions");
                    $this->reportLogsRepository->save($this->createReportLog("Not translating category $categoryId ({$category->getName()}) for store {$category->getStore()->getId()} because of failed conditions"));
                }

                $queue->acknowledge($message);
            } catch (Throwable $e) {
                $this->logger->error('Error updating category with ID ' . ($categoryId ?? 'N/A') . ' in store ' . ($storeId ?? 'N/A') . ': ' . $e->getMessage());
                $queue->reject($message, true, $e->getMessage());
                sleep(5); // Sleep to prevent retry storm
            }
        } else {
            usleep(100_000);
        }
    }

    /**
     * @param CategoryInterface $category
     * @param $storeId
     * @return CategoryInterface
     * @throws CouldNotSaveException
     */
    private function updateCategoryAttributes($category, $storeId)
    {
        if ($this->configManager->isEnabledCategoryUrlKeyTranslation($storeId)) {
            $store = $this->storeManager->getStore($storeId);
            $newUrlKey = $this->translateUrlKey($category->getUrlKey(), $store);
            if ($newUrlKey !== $category->getUrlKey()) {
                $category->addData([
                    'url_key' => $newUrlKey,
                    'exclude_clonable_auto_translation'=> true,
                ]);
                return $this->categoryRepository->save($category);
            }
        }

        return $category;
    }

    /**
     * @param CategoryInterface $category
     * @param $storeId
     * @return void
     * @throws Exception
     */
    private function updateUrlRewrite($category, $storeId): void
    {
        if ($this->configManager->isEnabledCategoryUrlKeyTranslation($storeId)) {
            $category->setData('save_rewrites_history', true);
            $urls = $this->categoryUrlRewriteGenerator->generate($category);
            try {
                $this->urlPersist->replace($urls);
            } catch (UrlAlreadyExistsException $e) {
                $this->updateAlreadyExistsUrl($category, $storeId);
            } catch (Exception $e) {
                throw new Exception($e->getMessage());
            }
        }
    }

    /**
     * @param CategoryInterface $category
     * @param $storeId
     * @throws Exception
     */
    private function updateAlreadyExistsUrl($category, $storeId): void
    {
        $category->setUrlKey($category->getUrlKey() . '-' . $category->getId());
        $category = $this->categoryRepository->save($category);
        $this->updateUrlRewrite($this->categoryRepository->get($category->getId()), $storeId);
    }
}
