<?php

namespace Clonable\Translator\Model\Product;

use Clonable\Translator\Api\Data\Product\TranslatorProductInterface;
use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Api\Service\ClonableTranslatorApiInterface;
use Clonable\Translator\Model\AbstractTranslator;
use Clonable\Translator\Model\ConfigManager;
use Clonable\Translator\Model\Logger\Logger;
use Clonable\Translator\Model\Product\Condition\ConditionChain;
use Clonable\Translator\Model\ReportLogsFactory;
use Exception;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Attribute\ScopeOverriddenValue;
use Magento\Catalog\Model\ResourceModel\Product\Action;
use Magento\CatalogUrlRewrite\Model\ProductUrlRewriteGenerator;
use Magento\Framework\App\CacheInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\MessageQueue\QueueInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\StoreManagerInterface;
use Magento\UrlRewrite\Model\Exception\UrlAlreadyExistsException;
use Magento\UrlRewrite\Model\UrlFinderInterface;
use Magento\UrlRewrite\Model\UrlPersistInterface;
use Throwable;


class TranslatorProduct extends AbstractTranslator implements TranslatorProductInterface {
    /**
     * @var ProductRepositoryInterface
     */
    private ProductRepositoryInterface $productRepository;


    /**
     * @var ProductUrlRewriteGenerator
     */
    private ProductUrlRewriteGenerator $productUrlRewriteGenerator;

    /**
     * @var Action
     */
    private Action $action;

    /**
     * @var Logger
     */
    private Logger $logger;

    /**
     * @var UrlPersistInterface
     */
    protected UrlPersistInterface $urlPersist;

    /**
     * @var StoreManagerInterface
     */
    protected StoreManagerInterface $storeManager;

    /**
     * @var ConfigManager
     */
    protected ConfigManager $configManager;

    private ReportLogsRepositoryInterface $reportLogsRepository;

    private ScopeOverriddenValue $scopeOverriddenValue;

    private ConditionChain $conditionChain;

    public function __construct(
        UrlPersistInterface            $urlPersist,
        UrlFinderInterface             $urlFinder,
        CacheInterface                 $cache,
        ProductRepositoryInterface     $productRepository,
        ProductUrlRewriteGenerator     $productUrlRewriteGenerator,
        Action                         $action,
        Json                           $json,
        Logger                         $logger,
        StoreManagerInterface          $storeManager,
        ConfigManager                  $configManager,
        ClonableTranslatorApiInterface $clonableTranslatorAPIInterface,
        ReportLogsRepositoryInterface  $reportLogsRepository,
        ReportLogsFactory              $reportLogsFactory,
        ScopeOverriddenValue $scopeOverriddenValue,
        ConditionChain       $conditionChain
    ) {
        parent::__construct($urlPersist, $urlFinder, $cache, $json, $storeManager, $configManager, $clonableTranslatorAPIInterface, $reportLogsFactory);
        $this->productRepository = $productRepository;
        $this->productUrlRewriteGenerator = $productUrlRewriteGenerator;
        $this->action = $action;
        $this->logger = $logger;
        $this->reportLogsRepository = $reportLogsRepository;
        $this->reportLogsFactory = $reportLogsFactory;
        $this->scopeOverriddenValue = $scopeOverriddenValue;
        $this->conditionChain = $conditionChain;
    }

    /**
     * @param QueueInterface $queue
     * @return void
     **/
    public function processMessage(QueueInterface $queue): void {
        if ($message = $queue->dequeue()) {
            try {
                $data = $this->json->unserialize($this->json->unserialize($message->getBody()));
                if (!is_array($data) || !array_key_exists('productId', $data) || !array_key_exists('storeId', $data) || !array_key_exists('force', $data)) {
                    $this->logger->error("Invalid data: " . print_r($data, true));
                    $queue->reject($message, false, "Invalid data");
                    return;
                }

                $productId = intval($data['productId']);
                $storeId = intval($data['storeId']);
                $force = $data['force'];

                if ($productId == 0 || $storeId == 0) {
                    $this->logger->warning('No store IDs available for translation product SKU ' . $productId);
                    $this->reportLogsRepository->save($this->createReportLog('No store IDs available for translation product SKU ' . $productId));
                    $queue->reject($message, false, "Invalid data");
                    return;
                }

                // Retrieve the base product for translations
                try {
                    $default_product = $this->getProductById($productId);
                    $product = $this->getProductById($productId, $storeId);
                } catch (NoSuchEntityException $e) {
                    $this->logger->error("Failed to load (base) product $productId for store $storeId");
                    $this->reportLogsRepository->save($this->createReportLog("Failed to load (base) product $productId for store $storeId"));
                    $queue->reject($message, false, "Non-existent product $productId for store $storeId");
                    return;
                }

                // Final check
                if ($this->conditionChain->shouldProcess($product, $force)) {
                    $this->logger->info("Translating product $productId ({$product->getName()}) for store {$product->getStore()->getId()}");

                    $startScriptTime = microtime(true);
                    $this->updateProductAttributes($product, $default_product);
                    $this->updateUrlRewrite($product);
                    $this->updateExcludeClonableAutoTranslation($product->getId(), true, $product->getStoreId());

                    $totalTime = microtime(true) - $startScriptTime;
                    $this->logger->info("Translated product {$productId} in store {$storeId}: {$totalTime} seconds");
                } else {
                    $this->logger->info("Not translating product $productId ({$product->getName()}) for store {$product->getStore()->getId()} because of failed conditions");
                    $this->reportLogsRepository->save($this->createReportLog("Not translating product $productId ({$product->getName()}) for store {$product->getStore()->getId()} because of failed conditions"));
                }

                $queue->acknowledge($message);
            } catch (Throwable $e) {
                $this->logger->error('Error updating product with ID ' . ($productId ?? 'N/A') . ' in store ' . ($storeId ?? 'N/A') . ': ' . $e->getMessage());
                $queue->reject($message, true, $e->getMessage());
                sleep(5); // Sleep to prevent retry storm
            }
        } else {
            usleep(100_000);
        }
    }

    /**
     * @param ProductInterface $product
     * @param ProductInterface $defaultProduct
     * @throws Exception
     */
    private function updateProductAttributes($product, $defaultProduct) {
        try {
            $startTranslateTime = microtime(true);
            $product_name = $defaultProduct->getName();

            $translatedText = $this->translateText($product_name, $product->getStore());
            $updatedAttributes = ['name' => $translatedText];
            if ($this->configManager->isEnabledProductUrlKeyTranslation($product->getStoreId())) {
                $new_url = $this->getNewUrl($product, $translatedText);
                if ($new_url !== $product->getUrlKey()) {
                    $updatedAttributes ['url_key'] = $new_url;
                }
            }
            $endTranslateTime = microtime(true);
            $startUpdateAttrTime = microtime(true);
            $this->action->updateAttributes([$product->getId()], $updatedAttributes, $product->getStoreId());
            $endUpdateAttrTime = microtime(true);
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
        return [
            'end_translate_time' => $endTranslateTime - $startTranslateTime,
            'end_update_attr_time' => $endUpdateAttrTime - $startUpdateAttrTime
        ];
    }

    /**
     * @param ProductInterface $product
     * @param string $translatedText
     * @return string
     */
    private function getNewUrl($product, $translatedText) {
        if ($this->configManager->useProductNameBasedUrlKeyTranslation($product->getStoreId())) {
            $url_key = transliterator_transliterate('Any-Latin; Latin-ASCII; Lower()', $translatedText);
            $without_space = str_replace(" ", "-", $url_key);
            return preg_replace('/[^a-z0-9\-]/u', '', $without_space);
        }

        return $this->translateUrlKey($product->getUrlKey(), $product->getStore());
    }

    /**
     * @param $product
     * @throws Exception
     */
    private function updateUrlRewrite($product) {
        if ($this->configManager->isEnabledProductUrlKeyTranslation($product->getStoreId())) {
            $startUrlRewriteTime = microtime(true);
            $product->setData('save_rewrites_history', true);
            $urls = $this->productUrlRewriteGenerator->generate($product);
            try {
                $this->urlPersist->replace($urls);
            } catch (UrlAlreadyExistsException $e) {
                $this->updateAlreadyExistsUrl($product);
            } catch (Exception $e) {
                throw new Exception($e->getMessage());
            }
            $endUrlRewriteTime = microtime(true);
            return $endUrlRewriteTime - $startUrlRewriteTime;
        }
        return null;
    }

    /**
     * @throws Exception
     */
    private function updateAlreadyExistsUrl(ProductInterface $product): void {
        $this->action->updateAttributes([$product->getId()], ['url_key' => $product->getUrlKey() . '-' . $product->getId()], $product->getStoreId());
        $this->updateUrlRewrite($this->getProductById($product->getId()));
    }

    /**
     * @param $sku
     * @param null $storeId
     * @return ProductInterface
     * @throws NoSuchEntityException
     */
    private function getProductById($id, $storeId = null): ProductInterface {
        return $this->productRepository->getById($id, true, $storeId, true);
    }

    /**
     * @throws Exception
     */
    public function updateExcludeClonableAutoTranslation($productId, $exclude, $storeId): void {
        try {
            $this->action->updateAttributes(
                [$productId],
                ['exclude_clonable_autotranslation' => $exclude],
                $storeId
            );
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }
}
