<?php

namespace Clonable\Translator\Controller\Adminhtml\Category;

use Clonable\Translator\Model\Logger\Logger;
use Clonable\Translator\Model\Category\Condition\ConditionChain;
use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Catalog\Helper\Category as CategoryHelper;
use Magento\Catalog\Model\CategoryRepository;
use Magento\Catalog\Model\ResourceModel\Category\Collection;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\StoreManagerInterface;

class TranslateCategories extends Action
{
    private Logger $logger;
    private CategoryHelper $categoryHelper;
    private CategoryRepository $categoryRepository;
    private ConditionChain $conditionChain;
    private StoreManagerInterface $storeManager;

    public function __construct(
        Context $context,
        Logger $logger,
        CategoryHelper $categoryHelper,
        CategoryRepositoryInterface $categoryRepository,
        ConditionChain $conditionChain,
        StoreManagerInterface $storeManager
    )
    {
        parent::__construct($context);
        $this->logger = $logger;
        $this->categoryHelper = $categoryHelper;
        $this->categoryRepository = $categoryRepository;
        $this->conditionChain = $conditionChain;
        $this->storeManager = $storeManager;
        $this->force = false;
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        try {
            $force_param = $this->getRequest()->getParam('force');
            $store_group_id = $this->storeManager->getStore()->getStoreGroupId();
            $root_category_id = $this->storeManager->getGroup($store_group_id)->getRootCategoryId();

            /**
             * Retrieve all the categories in Magento
             * @var Collection $categories
             */
            $categories = $this->categoryHelper->getStoreCategories(false, true);
            foreach ($categories as $category) {
                /** @var CategoryInterface $category */
                if ($category->getParentId() == $root_category_id) {
                    // Only start translation from the main categories that are a direct child of the root category.
                    $this->runForAllStores($category, ($force_param == 1));
                }
            }
        } catch (\Throwable $exception) {
            $this->logger->error($exception->getMessage());
        }
    }

    /**
     * Retrieves all the stores for the category and will try to queue them for translation.
     *
     * @param CategoryInterface $category a main category
     * @param bool $force whether to force the translations (will ignore the switch)
     * @return void
     */
    private function runForAllStores(CategoryInterface $category, bool $force) {
        try {
            if ($category->getStore()->isDefault()) {
                foreach ($category->getStoreIds() as $storeId) {
                    try {
                        $this->prepareForTranslationQueue($category, $storeId, $force, true);
                    } catch (NoSuchEntityException $_) {
                        continue;
                    }
                }
            }
        } catch (NoSuchEntityException $e) {
            $this->logger->error($e->getMessage());
        }
    }

    /**
     * This method will add the url of the category to the translation queue and recursively
     * go through all of its children and their descendants.
     *
     * Will automatically force the re-translation of the parent category, this will ensure
     * that the rewrites are always updated correctly throughout the entire rewrite tree.
     *
     * @param CategoryInterface $category the category to translate
     * @param int|null $storeId the id of the store
     * @param bool $force whether to force the translations (will ignore the switch)
     * @param bool $isMainCategory whether the category is a main category or not
     * @return bool
     * @throws NoSuchEntityException
     */
    private function prepareForTranslationQueue(CategoryInterface $category, int $storeId, bool $force, bool $isMainCategory = false) {
        // First loop through the children before translating the main category
        $has_translated_children = false;
        if ($category->getChildren() !== null) {
            foreach (explode(',', $category->getChildren()) as $child_id) {
                try {
                    $child = $this->categoryRepository->get($child_id, $storeId);
                    if($this->prepareForTranslationQueue($child, $storeId, $force)) {
                        $has_translated_children = true;
                    }
                } catch (NoSuchEntityException $_) {
                    continue; // no entity, no worries
                }
            }
        }

        if ($this->conditionChain->shouldProcess($category, $force)) {
            $parentCategory = $this->categoryRepository->get($category->getParentId(), $storeId);
            $categoryByStore = $this->categoryRepository->get($category->getId(), $storeId);

            if ($parentCategory->getExcludeClonableAutoTranslation()) {
                // If main (or parent) has been translated, then we should create the rewrites for the current child.
                // This makes sure that new subcategories get translated correctly.
                $this->conditionChain->process($categoryByStore, $force, true);
            } else if ($isMainCategory && $has_translated_children) {
                // By translating the main category last,
                // we'll ensure that rewrites for all of its children are created correctly.
                $this->conditionChain->process($categoryByStore, $force, true);
            } else {
                // Otherwise, translate the category without forcing rewrites.
                $this->conditionChain->process($categoryByStore, $force);
            }
            return true;
        }
        return false;
    }
}