<?php

namespace Clonable\Translator\Model\Category;

use Clonable\Translator\Api\Data\Category\TranslatorCategoryInterface;
use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Api\Service\ClonableTranslatorApiInterface;
use Clonable\Translator\Model\AbstractTranslator;
use Clonable\Translator\Model\ConfigManager;
use Clonable\Translator\Model\Logger\Logger;
use Clonable\Translator\Model\Category\Condition\ConditionChain;
use Clonable\Translator\Model\ReportLogsFactory;
use Exception;
use Magento\Backend\Model\UrlInterface;
use Magento\Catalog\Api\CategoryRepositoryInterface;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\CatalogUrlRewrite\Model\CategoryUrlRewriteGenerator;
use Magento\Framework\App\CacheInterface;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\MessageQueue\QueueInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\StoreManagerInterface;
use Magento\UrlRewrite\Model\Exception\UrlAlreadyExistsException;
use Magento\UrlRewrite\Model\UrlFinderInterface;
use Magento\UrlRewrite\Model\UrlPersistInterface;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite;
use Throwable;

class TranslatorCategory extends AbstractTranslator implements TranslatorCategoryInterface
{
    private CategoryRepositoryInterface $categoryRepository;
    private CategoryUrlRewriteGenerator $categoryUrlRewriteGenerator;
    private ReportLogsRepositoryInterface $reportLogsRepository;
    private ConditionChain $conditionChain;
    protected UrlFinderInterface $urlFinder;
    private Logger $logger;

    public function __construct(
        UrlPersistInterface            $urlPersist,
        UrlFinderInterface             $urlFinder,
        CacheInterface                 $cache,
        CategoryRepositoryInterface    $categoryRepository,
        CategoryUrlRewriteGenerator    $categoryUrlRewriteGenerator,
        Json                           $json,
        Logger                         $logger,
        StoreManagerInterface          $storeManager,
        ConfigManager                  $configManager,
        ClonableTranslatorApiInterface $clonableTranslatorAPIInterface,
        ReportLogsRepositoryInterface  $reportLogsRepository,
        ReportLogsFactory              $reportLogsFactory,
        ConditionChain       $conditionChain
    ) {
        parent::__construct($urlPersist, $urlFinder, $cache, $json, $storeManager, $configManager, $clonableTranslatorAPIInterface, $reportLogsFactory);
        $this->categoryRepository = $categoryRepository;
        $this->categoryUrlRewriteGenerator = $categoryUrlRewriteGenerator;
        $this->logger = $logger;
        $this->reportLogsRepository = $reportLogsRepository;
        $this->conditionChain = $conditionChain;
        $this->urlFinder = $urlFinder;
    }

    public function processMessage(QueueInterface $queue): void
    {
        if ($message = $queue->dequeue()) {
            try {
                $data = $this->json->unserialize($this->json->unserialize($message->getBody()));
                if (!is_array($data) || !array_key_exists('categoryId', $data) || !array_key_exists('storeId', $data) || !array_key_exists('force', $data)) {
                    $this->logger->error("Invalid data: " . print_r($data, true));
                    $queue->reject($message, false, "Invalid data");
                    return;
                }

                // Prepare and parse serialized data
                $categoryId = intval($data['categoryId']);
                $storeId = intval($data['storeId']);
                $force = $data['force'];
                $createRewrites = ($data['createRewrites'] ?? false);

                if ($categoryId == 0 || $storeId == 0) {
                    $this->logger->warning('No store IDs available for translation category ID ' . $categoryId);
                    $this->reportLogsRepository->save($this->createReportLog('No store IDs available for translation category SKU ' . $categoryId));
                    $queue->reject($message, false, "Invalid data");
                    return;
                }

                $this->storeManager->setCurrentStore($storeId);
                // Retrieve the base category for translations
                try {
                    $category = $this->categoryRepository->get($categoryId, $storeId);
                } catch (NoSuchEntityException $e) {
                    $this->logger->error("Failed to load category $categoryId for store $storeId");
                    $this->reportLogsRepository->save($this->createReportLog("Failed to load category $categoryId for store $storeId"));
                    $queue->reject($message, false, "Non-existent category $categoryId for store $storeId");
                    return;
                }

                if ($this->conditionChain->shouldProcess($category, $force)) {
                    $this->logger->info("Translating category $categoryId ({$category->getName()}) for store {$category->getStore()->getId()}");

                    $startScriptTime = microtime(true);
                    $category = $this->updateCategoryAttributes($category, $storeId);
                    if ($createRewrites) {
                        $this->updateUrlRewrite($category, $storeId);
                    }

                    $totalTime = microtime(true) - $startScriptTime;
                    $this->logger->info("Translated category {$categoryId} in store {$storeId}: {$totalTime} seconds");
                } else {
                    $this->logger->info("Not translating category $categoryId ({$category->getName()}) for store {$category->getStore()->getId()} because of failed conditions");
                    $this->reportLogsRepository->save($this->createReportLog("Not translating category $categoryId ({$category->getName()}) for store {$category->getStore()->getId()} because of failed conditions"));
                }

                $queue->acknowledge($message);
            } catch (Throwable $e) {
                $this->logger->error('Error updating category with ID ' . ($categoryId ?? 'N/A') . ' in store ' . ($storeId ?? 'N/A') . ': ' . $e->getMessage());
                $queue->reject($message, true, $e->getMessage());
                sleep(1); // Sleep to prevent retry storm
            }
        } else {
            usleep(100_000);
        }
    }

    /**
     * @param CategoryInterface $category
     * @param $storeId
     * @return CategoryInterface
     * @throws CouldNotSaveException
     */
    private function updateCategoryAttributes(CategoryInterface $category, int $storeId)
    {
        if ($this->configManager->isEnabledCategoryUrlKeyTranslation($storeId)) {
            $store = $this->storeManager->getStore($storeId);
            $newUrlKey = $this->translateUrlKey($category->getUrlKey(), $store);
            if ($newUrlKey !== $category->getUrlKey()) {
                $category->addData(['url_key' => $newUrlKey]);
            }
            // Make sure to always toggle the auto translation;
            // This will make sure that translations that are the same as the original get skipped.
            $category->addData([
                'exclude_clonable_auto_translation'=> true,
            ]);
            return $this->categoryRepository->save($category);
        }

        return $category;
    }

    /**
     * Update the url rewrites for the category.
     * When the category has child categories, the rewrites for those categories will be created.
     *
     * @param CategoryInterface $category
     * @param int $storeId
     * @return void
     * @throws CouldNotSaveException
     * @throws NoSuchEntityException
     */
    private function updateUrlRewrite(CategoryInterface $category, int $storeId): void
    {
        if ($this->configManager->isEnabledCategoryUrlKeyTranslation($storeId)) {
            $category->setData('save_rewrites_history', true);
            $store_group_id = $this->storeManager->getStore()->getStoreGroupId();
            $root_category_id = $this->storeManager->getGroup($store_group_id)->getRootCategoryId();

            try {
                // Make sure to only create the rewrite once for the main category;
                // This will ensure that rewrites are only created once.
                $urls = $this->categoryUrlRewriteGenerator->generate($category, true, $root_category_id);
                $urls = $this->rewriteUrls($category, $urls, $storeId, $root_category_id);

                if (!empty($urls)) {
                    $this->urlPersist->replace($urls);
                }
            } catch (UrlAlreadyExistsException $e) {
                $this->updateAlreadyExistsUrl($category, $storeId);
            } catch (Exception $e) {
                // Probably a [Unique constraint violation found]
                // Shouldn't be happening but is probably due to some manual tinkering with the translations.
                $this->reportLogsRepository->save($this->createReportLog("Failed to update url rewrite for store $storeId"));
            }
        }
    }

    /**
     * Create a map of translated url keys of the category and all of its children.
     *
     * @param int $category_id the id of the category
     * @param int $store_id the id of the store
     * @return array
     * @throws NoSuchEntityException
     */
    private function createUrlKeyTranslationMap(int $category_id, int $store_id): array {
        $category = $this->categoryRepository->get($category_id, $store_id);
        $default_category = $this->categoryRepository->get($category_id, $this->storeManager->getDefaultStoreView()->getId());

        $map = [$default_category->getUrlKey() => $category->getUrlKey()];
        if ($category->getChildren()) {
            foreach (explode(',', $category->getChildren()) as $child) {
                $map = array_merge($map, $this->createUrlKeyTranslationMap(intval($child), $store_id));
            }
        }
        return $map;
    }

    /**
     * Apply a transform on a path.
     * The method will split the parts on the slash, and tries to replace the parts with the translated variant.
     *
     * @param string $path the path that needs to be transformed
     * @param array $urlKeyTranslationMap the map of with the translated parts
     * @return string
     */
    private function transformPath(string $path, array $urlKeyTranslationMap): string {
        $path_parts = explode('/', $path);
        $new_path_parts = [];
        foreach ($path_parts as $part) {
            if (key_exists($part, $urlKeyTranslationMap)) {
                $new_path_parts[] = $urlKeyTranslationMap[$part];
            } else {
                $new_path_parts[] = $part;
            }
        }
        return implode('/', $new_path_parts);
    }

    /**
     * Rewrite all the auto-generated urls of Magento.
     * This will ensure that the translated url keys will be used for the rewrites.
     *
     * @param CategoryInterface $category the category to create rewrites for
     * @param UrlRewrite[] $urlRewrites list of generated urls rewrites to modify
     * @param int $storeId the id of store
     * @param int $rootCategoryId the root category of the store
     * @return UrlRewrite[]
     * @throws NoSuchEntityException
     */
    private function rewriteUrls(CategoryInterface $category, array $urlRewrites, int $storeId, int $rootCategoryId): array {
        $root_category = $this->categoryRepository->get($rootCategoryId);
        $url_key_map = $this->createUrlKeyTranslationMap($root_category->getId(), $storeId);

        $rewrites = [];
        foreach ($urlRewrites as $url) {
            if ($url->getRequestPath() === $category->getUrlKey() && $url->getRedirectType() === 301) {
                // specific case for reversing the redirect to itself.
                $url->setRequestPath($url->getTargetPath());
                $url->setTargetPath($category->getUrlKey());
            } else if ($url->getRedirectType() === 0) {
                $new_request_path = $this->transformPath($url->getRequestPath(), $url_key_map);
                $url->setRequestPath($new_request_path);
            } else if ($url->getRedirectType() === 301) {
                $new_target_path = $this->transformPath($url->getTargetPath(), $url_key_map);
                $url->setTargetPath($new_target_path);
            }
            $rewrites[] = $url;
        }


        // Debug log rewrites. Can be useful for future debugging
        foreach ($rewrites as $rewrite) {
            $t = $rewrite->getTargetPath();
            $r = $rewrite->getRequestPath();
            $redirect = $rewrite->getRedirectType();
            $this->logger->debug("[$redirect] $r --> $t");
        }

        return $rewrites;
    }

    /**
     * @param CategoryInterface $category
     * @param $storeId
     * @throws NoSuchEntityException|CouldNotSaveException
     */
    private function updateAlreadyExistsUrl($category, $storeId): void
    {
        $category->setUrlKey($category->getUrlKey() . '-' . $category->getId());
        $category = $this->categoryRepository->save($category);
        $this->updateUrlRewrite($this->categoryRepository->get($category->getId()), $storeId);
    }
}
