<?php

namespace Clonable\Translator\Controller\Adminhtml\Category;

use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Model\Logger\Logger;
use Clonable\Translator\Model\Category\Condition\ConditionChain;
use Clonable\Translator\Model\ReportLogs;
use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Catalog\Model\ResourceModel\Category\Collection;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Clonable\Translator\Model\ReportLogsFactory;

class TranslateCategories extends Action
{
    private Logger $logger;
    private CategoryCollectionFactory $categoryCollectionFactory;
    private ConditionChain $conditionChain;
    private RequestInterface $request;
    private StoreManagerInterface $storeManager;
    private ReportLogsRepositoryInterface $reportLogsRepository;
    protected ReportLogsFactory $reportLogsFactory;

    public function __construct(
        Context                       $context,
        Logger                        $logger,
        CategoryCollectionFactory     $categoryCollectionFactory,
        ConditionChain                $conditionChain,
        RequestInterface              $request,
        ReportLogsRepositoryInterface $reportLogsRepository,
        StoreManagerInterface         $storeManager,
        ReportLogsFactory             $reportLogsFactory
    )
    {
        parent::__construct($context);
        $this->logger = $logger;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
        $this->conditionChain = $conditionChain;
        $this->request = $request;
        $this->storeManager = $storeManager;
        $this->reportLogsRepository = $reportLogsRepository;
        $this->reportLogsFactory = $reportLogsFactory;
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        try {
            $force_param = $this->request->getParam('force');
            $store_id = $this->getRequest()->getParam('store', Store::DEFAULT_STORE_ID);

            try {
                $store = $this->storeManager->getStore($store_id);
            } catch (NoSuchEntityException $e) {
                $this->reportLogsRepository->save(
                    $this->createReportLog("Cannot find store id with id: $store_id", Store::DEFAULT_STORE_ID)
                );
                return;
            }

            /**
             * Retrieve the categories to be translated.
             * @var Collection $collection
             */
            $categories = $this->categoryCollectionFactory->create()
                ->setStoreId($store->getId())
                ->addAttributeToSelect('url_key')
                ->addFieldToFilter('entity_id', ['nin' => [1, $store->getRootCategoryId()]]); // remove the root category

            foreach ($categories as $category) {
                /** @var CategoryInterface $category */
                $category->setStoreId($store->getId());
                $this->conditionChain->process($category, ($force_param == 1));
            }
        } catch (\Throwable $exception) {
            $this->logger->error("Execute TranslateCategories: " . $exception->getMessage());
        }
    }

    private function createReportLog(string $message, int $store_id): ReportLogs
    {
        $reportLog = $this->reportLogsFactory->create();
        $reportLog->setMessage($message);
        $reportLog->setStoreId($store_id);
        return $reportLog;
    }
}