<?php

namespace Clonable\Translator\Model\Product;

use Clonable\Translator\Api\Data\Product\TranslatorProductInterface;
use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Api\Service\ClonableTranslatorApiInterface;
use Clonable\Translator\Model\AbstractTranslator;
use Clonable\Translator\Model\ConfigManager;
use Clonable\Translator\Model\Logger\Logger;
use Clonable\Translator\Model\Product\Condition\ConditionChain;
use Clonable\Translator\Model\ReportLogsFactory;
use Exception;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Attribute\ScopeOverriddenValue;
use Magento\Catalog\Model\ResourceModel\Product\Action;
use Magento\CatalogUrlRewrite\Model\ProductUrlRewriteGenerator;
use Magento\Framework\App\CacheInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\MessageQueue\QueueInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use Magento\UrlRewrite\Model\Exception\UrlAlreadyExistsException;
use Magento\UrlRewrite\Model\UrlFinderInterface;
use Magento\UrlRewrite\Model\UrlPersistInterface;
use Magento\UrlRewrite\Service\V1\Data\UrlRewriteFactory as UrlRewriteDataFactory;
use Magento\UrlRewrite\Model\ResourceModel\UrlRewriteCollectionFactory as RewriteCollectionFactory;
use Throwable;

class TranslatorProduct extends AbstractTranslator implements TranslatorProductInterface {

    private ProductRepositoryInterface $productRepository;
    private ProductUrlRewriteGenerator $productUrlRewriteGenerator;
    private Action $action;
    private Logger $logger;
    protected UrlPersistInterface $urlPersist;
    protected StoreManagerInterface $storeManager;

    protected ConfigManager $configManager;
    protected ReportLogsRepositoryInterface $reportLogsRepository;
    private ConditionChain $conditionChain;
    protected ReportLogsFactory $reportLogsFactory;

    public function __construct(
        UrlPersistInterface            $urlPersist,
        UrlFinderInterface             $urlFinder,
        CacheInterface                 $cache,
        ProductRepositoryInterface     $productRepository,
        ProductUrlRewriteGenerator     $productUrlRewriteGenerator,
        Action                         $action,
        Json                           $json,
        Logger                         $logger,
        StoreManagerInterface          $storeManager,
        ScopeConfigInterface           $scopeConfig,
        ConfigManager                  $configManager,
        ClonableTranslatorApiInterface $clonableTranslatorAPIInterface,
        ReportLogsRepositoryInterface  $reportLogsRepository,
        UrlRewriteDataFactory          $urlRewriteFactory,
        RewriteCollectionFactory       $rewriteCollectionFactory,
        ReportLogsFactory              $reportLogsFactory,
        ConditionChain                 $conditionChain
    ) {
        parent::__construct($urlPersist, $urlFinder, $cache, $json, $storeManager, $configManager, $clonableTranslatorAPIInterface, $scopeConfig, $urlRewriteFactory, $rewriteCollectionFactory, $reportLogsFactory);
        $this->productRepository = $productRepository;
        $this->productUrlRewriteGenerator = $productUrlRewriteGenerator;
        $this->action = $action;
        $this->logger = $logger;
        $this->reportLogsRepository = $reportLogsRepository;
        $this->reportLogsFactory = $reportLogsFactory;
        $this->conditionChain = $conditionChain;
    }

    private function print(string $message, string $type = 'error')
    {
        if ($type == 'error') {
            $this->logger->error($message);
            $this->reportLogsRepository->save($this->createReportLog($message));
        } else if ($type === 'warning') {
            $this->logger->warning($message);
        } else if ($type === 'info') {
            $this->logger->info($message);
        } else if ($type === 'debug') {
            $this->logger->debug($message);
        }
    }

    /**
     * @param QueueInterface $queue
     * @return void
     **/
    public function processMessage(QueueInterface $queue): void {
        if ($message = $queue->dequeue()) {
            try {
                $data = $this->json->unserialize($this->json->unserialize($message->getBody()));
                if (!is_array($data) || !array_key_exists('productId', $data) || !array_key_exists('storeId', $data) || !array_key_exists('force', $data)) {
                    $this->print("Invalid data: " . print_r($data, true));
                    $queue->reject($message, false, "Invalid data");
                    return;
                }

                $product_id = intval($data['productId']);
                $store_id = intval($data['storeId']);
                $force = boolval($data['force']);

                if ($product_id == 0 || $store_id == 0) {
                    $this->print('No store IDs available for translation product ID ' . $product_id, 'warning');
                    $queue->reject($message, false, "Invalid data");
                    return;
                }

                // Retrieve the base product for translations
                try {
                    $store = $this->storeManager->getstore($store_id);
                    $default_product = $this->getProductById($product_id);
                    $product = $this->getProductById($product_id, $store_id);
                } catch (NoSuchEntityException $e) {
                    $this->print("Failed to load product $product_id for store $store_id");
                    $queue->reject($message, false, "Non-existent product $product_id for store $store_id");
                    return;
                }

                if ($this->conditionChain->shouldProcess($product, $force)) {
                    // Update the product data (stuff like url key and product name)
                    $this->updateProductAttributes($product, $default_product);
                    // Make sure to store the rewrites for the product data.
                    $this->createTranslatedUrlRewrite($product, $store, $default_product);
                    // Save exclude from Clonable auto translation separately (don't really know why)
                    $this->updateExcludeClonableAutoTranslation($product->getId(), true, $product->getStoreId());
                }

                $queue->acknowledge($message);
            } catch (Throwable $e) {
                $this->print('Error updating product with ID ' . ($product_id ?? 'N/A') . ' in store ' . ($storeId ?? 'N/A') . ': ' . $e->getMessage());
                $queue->reject($message, true, $e->getMessage());
                sleep(1); // Sleep to prevent retry storm
            }
        } else {
            usleep(100_000);
        }
    }

    /**
     * @param ProductInterface $product
     * @param ProductInterface $defaultProduct
     * @throws Exception
     */
    private function updateProductAttributes(ProductInterface &$product, ProductInterface $defaultProduct) {
        $original_name = $defaultProduct->getName();
        $translated_name = $this->translateText($original_name, $product->getStore());
        $updated_attributes = ['name' => $translated_name];
        $product->setName($translated_name); // make sure to also set the reference

        if ($this->configManager->isEnabledProductUrlKeyTranslation($product->getStoreId())) {
            $new_url_key = $this->getNewUrl($product, $translated_name);
            if ($new_url_key !== $product->getUrlKey()) {
                $updated_attributes['url_key'] = $new_url_key;
                $product->setUrlKey($new_url_key); // make sure to also set the reference
            }
        }

        // save the changed attribute to the database.
        $this->action->updateAttributes([$product->getId()], $updated_attributes, $product->getStoreId());
    }

    /**
     * Build a URL‐safe key, either from the translated name or via the translator service.
     *
     * @param ProductInterface $product
     * @param string           $translated_text
     * @return string
     */
    private function getNewUrl(ProductInterface $product, string $translated_text) {
        if ($this->configManager->useProductNameBasedUrlKeyTranslation($product->getStoreId())) {
            $url_key = transliterator_transliterate('Any-Latin; Latin-ASCII; Lower()', $translated_text);
            $without_space = str_replace(" ", "-", $url_key);
            return preg_replace('/[^a-z0-9\-]/u', '', $without_space);
        }

        return $this->translateUrlKey($product->getUrlKey(), $product->getStore());
    }

    /**
     * Regenerate all URL‐rewrites for $product and then add a 301 redirect
     * from the old path → new path.
     *
     * @param ProductInterface $product Already loaded in target store
     * @param Store $store
     * @param ProductInterface $default_product
     * @return void
     * @throws Exception
     */
    private function createTranslatedUrlRewrite(ProductInterface $product, Store $store, ProductInterface $default_product) {
        if (!$this->configManager->isEnabledProductUrlKeyTranslation($product->getStoreId())) {
            return;
        }

        try {
            // Generate the rewrites, but do not persist them yet
            $rewrites = $this->productUrlRewriteGenerator->generate($product);

            // make sure to redirect the original to the translated
            $suffix = $this->getSuffix("product", $store);
            $original_path = $default_product->getUrlKey() . $suffix;
            $rewrite = $this->getRedirect($product, $store, $original_path, $rewrites);
            if ($rewrite) {
                $rewrites[] = $rewrite;
            }

            $this->urlPersist->replace($rewrites);
        } catch (UrlAlreadyExistsException $e) {
            $product->setData('url_key', $product->getUrlKey() . '-' . $product->getId());
        }
    }

    /**
     * @param int      $id
     * @param int|null $storeId
     * @return ProductInterface
     * @throws NoSuchEntityException
     */
    private function getProductById($id, $storeId = null): ProductInterface {
        return $this->productRepository->getById($id, true, $storeId, true);
    }

    /**
     * @param int      $productId
     * @param bool     $exclude
     * @param int|null $storeId
     * @throws Exception
     */
    public function updateExcludeClonableAutoTranslation($productId, $exclude, $storeId): void {
        try {
            $this->action->updateAttributes(
                [$productId],
                ['exclude_clonable_autotranslation' => $exclude],
                $storeId
            );
        } catch (Exception $e) {
            throw new Exception($e->getMessage());
        }
    }
}
