<?php

namespace Clonable\Translator\Model\Product;

use Clonable\Translator\Api\Data\Product\TranslatorProductInterface;
use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Api\Service\ClonableTranslatorApiInterface;
use Clonable\Translator\Exception\InvalidEntityTypeException;
use Clonable\Translator\Exception\ProductTranslationException;
use Clonable\Translator\Model\AbstractTranslator;
use Clonable\Translator\Model\ConfigManager;
use Clonable\Translator\Model\Logger\Logger;
use Clonable\Translator\Model\Product\Condition\ConditionChain;
use Clonable\Translator\Model\ReportLogsFactory;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory as ProductCollectionFactory;
use Magento\CatalogUrlRewrite\Model\ProductUrlRewriteGenerator;
use Magento\Framework\App\CacheInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Exception\StateException;
use Magento\Framework\Filter\FilterManager;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use Magento\UrlRewrite\Model\Exception\UrlAlreadyExistsException;
use Magento\UrlRewrite\Model\UrlFinderInterface;
use Magento\UrlRewrite\Model\UrlPersistInterface;
use Magento\UrlRewrite\Service\V1\Data\UrlRewriteFactory as UrlRewriteDataFactory;
use Magento\UrlRewrite\Model\ResourceModel\UrlRewriteCollectionFactory as RewriteCollectionFactory;
use Throwable;

class ProductTranslator extends AbstractTranslator implements TranslatorProductInterface {

    private ProductRepositoryInterface $productRepository;
    private ProductUrlRewriteGenerator $productUrlRewriteGenerator;
    private Logger $logger;
    protected UrlPersistInterface $urlPersist;
    protected StoreManagerInterface $storeManager;

    protected ConfigManager $configManager;
    private FilterManager $filterManager;
    protected ReportLogsRepositoryInterface $reportLogsRepository;
    private ConditionChain $conditionChain;
    protected ReportLogsFactory $reportLogsFactory;
    private ProductCollectionFactory $productCollectionFactory;

    public function __construct(
        UrlPersistInterface            $urlPersist,
        UrlFinderInterface             $urlFinder,
        CacheInterface                 $cache,
        ProductRepositoryInterface     $productRepository,
        ProductUrlRewriteGenerator     $productUrlRewriteGenerator,
        Json                           $json,
        Logger                         $logger,
        ScopeConfigInterface           $scopeConfig,
        StoreManagerInterface          $storeManager,
        ConfigManager                  $configManager,
        ClonableTranslatorApiInterface $clonableTranslatorAPIInterface,
        ReportLogsRepositoryInterface  $reportLogsRepository,
        FilterManager                  $filterManager,
        UrlRewriteDataFactory          $urlRewriteFactory,
        RewriteCollectionFactory       $rewriteCollectionFactory,
        ReportLogsFactory              $reportLogsFactory,
        ConditionChain                 $conditionChain,
        ProductCollectionFactory       $productCollectionFactory
    ) {
        parent::__construct($urlPersist, $urlFinder, $cache, $json, $storeManager, $configManager, $clonableTranslatorAPIInterface, $scopeConfig, $urlRewriteFactory, $rewriteCollectionFactory, $reportLogsFactory);
        $this->productRepository = $productRepository;
        $this->productUrlRewriteGenerator = $productUrlRewriteGenerator;
        $this->logger = $logger;
        $this->reportLogsRepository = $reportLogsRepository;
        $this->reportLogsFactory = $reportLogsFactory;
        $this->conditionChain = $conditionChain;
        $this->filterManager = $filterManager;
        $this->productCollectionFactory = $productCollectionFactory;
    }

    private function print(string $message, string $type = 'error')
    {
        if ($type == 'error') {
            $this->logger->error($message);
            $this->reportLogsRepository->save($this->createReportLog($message));
        } else if ($type === 'warning') {
            $this->logger->warning($message);
        } else if ($type === 'info') {
            $this->logger->info($message);
        } else if ($type === 'debug') {
            $this->logger->debug($message);
        }
    }

    /**
     * @inheritDoc
     * @throws ProductTranslationException
     **/
    public function translate(string $body): void {
        try {
            $data = $this->json->unserialize($body);
            if (!is_array($data) || !array_key_exists('productId', $data) || !array_key_exists('storeId', $data) || !array_key_exists('force', $data)) {
                $this->print("Invalid data: " . print_r($data, true));
                throw new ProductTranslationException(__("Invalid message queue data"));
            }

            $product_id = intval($data['productId']);
            $store_id = intval($data['storeId']);
            $force = boolval($data['force']);

            if ($product_id == 0 || $store_id == 0) {
                $this->print('No store IDs available for translation product ID ' . $product_id, 'warning');
                throw new ProductTranslationException(__("Invalid message queue data"));
            }

            // Retrieve the base product for translations
            try {
                $store = $this->storeManager->getstore($store_id);
                $product = $this->productRepository->getById($product_id, true, $store_id, true);
            } catch (NoSuchEntityException $e) {
                $this->print("Failed to load product $product_id for store $store_id");
                throw new ProductTranslationException(__("Non-existent product $product_id for store $store_id"));
            }

            if ($this->conditionChain->shouldProcess($product, $force)) {
                $this->translateProduct($product, $store);
            }
        } catch (InvalidEntityTypeException $exception) {
            $this->print("Cannot create redirects for an unsupported entity type:");
            throw new ProductTranslationException(__("Cannot create redirects for an unsupported entity type:"),null, 0, false);
        } catch (Throwable $e) {
            $this->print('Error updating product with ID ' . ($product_id ?? 'N/A') . ' in store ' . ($store_id ?? 'N/A') . ': ' . $e->getMessage());
            throw new ProductTranslationException(__($e->getMessage()),  null, 0, true);
        }
    }

    private function translateProduct(ProductInterface $product, Store $store): void {
        if ((int)$product->getStoreId() === Store::DEFAULT_STORE_ID || $store->getId() === Store::DEFAULT_STORE_ID) {
            $this->print("Product from the default store should not be translated.");
            return;
        }

        try {
            $default_product = $this->productRepository->getById($product->getId(), true, Store::DEFAULT_STORE_ID, true);
        } catch (NoSuchEntityException $e) {
            $this->print("Could not find the root product for {$product->getName()}: " . $e->getMessage());
            return;
        }

        // Update product name
        $original_name = $default_product->getName();
        $translated_name = $this->translateText($original_name, $product->getStore());
        $product->setName($translated_name);

        // CHeck and Update product URL
        if ($this->configManager->isEnabledProductUrlKeyTranslation($store->getId())) {
            $new_url_key = $this->getNewUrl($product, $translated_name, $store);
            $new_url_key = $this->ensureUniqueUrl($product, $new_url_key, $store);
            if ($new_url_key !== $product->getUrlKey()) {
                $product->setUrlKey($new_url_key);
            }
        }

        try {
            // IMPORTANT: this line is needed for saving the product to the correct store
            // otherwise the current store from the StoreManager would be used.
            // because it's not recommended to change the current store in the StoreManager, we need to rely on this line.
            $product->setStoreId($store->getId());
            $product->setData("exclude_clonable_autotranslation", true);
            $this->productRepository->save($product);
        } catch (CouldNotSaveException|StateException|InputException $e) {
            $this->print("{$product->getName()}({$product->getId()}) -> Could not save product after setting data:  " . $e->getMessage());
            return;
        } catch (Throwable $e) {

        }

        // Check whether URL rewrites should be created
        if ($this->configManager->isEnabledProductUrlKeyTranslation($product->getStoreId())) {
            try {
                $product = $this->productRepository->getById($product->getId(), true, $store->getId(), true);
                /**
                 * According to uncle GPT:
                 * After changing url_key, ensure Magento recalculates the product’s url_path before generating rewrites:
                 * You’re usually fine because the generator can compute it, but explicitly nulling avoids edge cases with stale values.
                 */
                $product->setData('url_path', null);

                // Generate the rewrites, but do not persist them yet
                $rewrites = $this->productUrlRewriteGenerator->generate($product);
                $suffix = $this->getSuffix("product", $store);

                // Double check if the original path does not already contain the url key
                $original_path = $default_product->getUrlKey();
                if ($suffix !== '' && substr_compare($original_path, $suffix, -strlen($suffix)) !== 0) {
                    $original_path .= $suffix;
                }

                // make sure to redirect the original to the translated
                $rewrite = $this->getRedirect($product, $store, $original_path, $rewrites);
                if ($rewrite) {
                    $rewrites[] = $rewrite;
                }

                $this->urlPersist->replace($rewrites);
            } catch (UrlAlreadyExistsException $e) {
                $this->print("{$product->getName()}({$product->getId()}) -> Duplicate url detected for product");
            } catch (\Throwable $e) {
                $this->print("{$product->getName()}({$product->getId()}) -> Unknown error occurred while persisting urls:  " . $e->getMessage());
            }
        }
    }

    /**
     * Build a URL‐safe key, either from the translated name or via the translator service.
     *
     * @param ProductInterface $product
     * @param string $translated_text
     * @param Store $store
     * @return string
     */
    private function getNewUrl(ProductInterface $product, string $translated_text, Store $store): string {
        if ($this->configManager->useProductNameBasedUrlKeyTranslation($product->getStoreId())) {
            return $this->filterManager->translitUrl($translated_text);
        }

        try {
            $default_product = $this->productRepository->getById($product->getId(), true, Store::DEFAULT_STORE_ID, true);
            return $this->translateUrlKey($default_product->getUrlKey(), $store);
        } catch (NoSuchEntityException $e) {
            return $product->getUrlKey();
        }
    }

    /**
     * Ensures that a new product key is unique in a given store.
     * If the key is unique, it returns the exact key as given in the $new_url_key parameter.
     * If the key is not unique, it will try to prefix a suffix beginning at -2 that increments until -9.
     * If the maximum prefix number is reached, it will return the original url key of the product, which results in a noop.
     *
     * @param ProductInterface $product
     * @param string $new_url_key the new url key you want to save
     * @param Store $store
     * @return string a save-to-save url key.
     */
    private function ensureUniqueUrl(ProductInterface $product, string $new_url_key, Store $store): string
    {
        $url_key_available = $this->isUrlKeyAvailable($new_url_key, $product, $store);
        if ($url_key_available) {
            // if its already available, then just return the new url.
            return $new_url_key;
        }
        $custom_suffix = 2;
        while (true) {
            $new_custom_url_key = "$new_url_key-$custom_suffix"; // prefix the custom suffix
            $url_key_available = $this->isUrlKeyAvailable($new_custom_url_key, $product, $store);
            if ($url_key_available) {
                // return the url key with the new suffix.
                return $new_custom_url_key;
            }
            $custom_suffix++; // don't forget increment
            if ($custom_suffix > 9) { // some break point, if this is reached, there's probably some really weird data.
                $this->print("Maximum number of suffixing reached within the product url key generator. You probably should inspect your product data.");
                break;
            }
        }
        // return the original url key, this should result in a noop when trying to save
        return $product->getUrlKey();
    }

    /**
     * Check whether a given url key is unique for a store.
     * @param string $url_key
     * @param ProductInterface $product
     * @param Store $store
     * @return bool
     */
    private function isUrlKeyAvailable(string $url_key, ProductInterface $product, Store $store): bool
    {
        $collection = $this->productCollectionFactory->create();
        $collection->setStoreId($store->getId());
        $collection->addAttributeToSelect('entity_id');
        $collection->addAttributeToFilter('url_key', $url_key);

        foreach ($collection as $item) {
            if ((int)$item->getId() !== $product->getId()) {
                return false;
            }
        }
        return true;
    }
}
