<?php

namespace Clonable\Translator\Model\Queue\Consumer;

use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Model\ReportLogs;
use Magento\Framework\MessageQueue\ConsumerConfigurationInterface;
use Magento\Framework\MessageQueue\ConsumerInterface;
use Magento\Framework\MessageQueue\EnvelopeInterface;
use Magento\Framework\MessageQueue\QueueInterface;
use Clonable\Translator\Model\ReportLogsFactory;
use Magento\Framework\Serialize\Serializer\Json;
use ReflectionClass;
use Throwable;

abstract class BaseConsumer implements ConsumerInterface
{
    private ConsumerConfigurationInterface $configuration;
    private ReportLogsRepositoryInterface $reportLogsRepository;
    private ReportLogsFactory $reportLogsFactory;
    protected Json $json;

    public function __construct
    (
        ConsumerConfigurationInterface $configuration,
        ReportLogsRepositoryInterface  $reportLogsRepository,
        ReportLogsFactory $reportLogsFactory,
        JSON $json
    ){
        $this->configuration = $configuration;
        $this->reportLogsRepository = $reportLogsRepository;
        $this->reportLogsFactory = $reportLogsFactory;
        $this->json = $json;
    }

    /**
     * @param int|null $maxNumberOfMessages
     */
    public function process($maxNumberOfMessages = null): void
    {
        $class = new ReflectionClass($this);
        $class_name = $class->getShortName();

        $queue = $this->configuration->getQueue();
        $max_number_of_messages = $maxNumberOfMessages ?? 1;
        $processed = 0;

        try {
            for ($i = 0; $i < $max_number_of_messages; $i++) {
                if ($message = $queue->dequeue()) {
                    $this->handleMessage($queue, $message);
                    $processed++;
                } else {
                    break;
                }
            }

            if ($processed > 0) {
                $processed_messages = number_format($processed);
                $this->reportLogsRepository->save($this->createReportLog("Successfully processed $class_name translations: total items = $processed_messages"));
            }
        } catch (Throwable $exception) {
            $this->reportLogsRepository->save($this->createReportLog("$class_name consumer error: {$exception->getMessage()}"));
        }
    }

    public abstract function handleMessage(QueueInterface $queue, EnvelopeInterface $message);

    /**
     *
     * @param string $message
     * @param string $storeId
     * @return ReportLogs
     */
    protected function createReportLog(string $message, string $storeId = '0'): ReportLogs {
        $reportLog = $this->reportLogsFactory->create();
        $reportLog->setMessage($message);
        $reportLog->setStoreId($storeId);
        return $reportLog;
    }
}