<?php

namespace Clonable\Translator\Controller\Adminhtml\Category;

use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Model\Logger\Logger;
use Clonable\Translator\Model\Category\Condition\ConditionChain;
use Clonable\Translator\Model\ReportLogs;
use Magento\Backend\App\Action;
use Magento\Backend\App\Action\Context;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Catalog\Model\ResourceModel\Category\Collection;
use Magento\Framework\App\RequestInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Catalog\Model\ResourceModel\Category\CollectionFactory as CategoryCollectionFactory;
use Clonable\Translator\Model\ReportLogsFactory;

class TranslateAllCategories extends Action
{
    private Logger $logger;
    private CategoryCollectionFactory $categoryCollectionFactory;
    private ConditionChain $conditionChain;
    private RequestInterface $request;
    private StoreManagerInterface $storeManager;
    private ReportLogsRepositoryInterface $reportLogsRepository;
    protected ReportLogsFactory $reportLogsFactory;

    public function __construct(
        Context                       $context,
        Logger                        $logger,
        CategoryCollectionFactory     $categoryCollectionFactory,
        ConditionChain                $conditionChain,
        RequestInterface              $request,
        ReportLogsRepositoryInterface $reportLogsRepository,
        StoreManagerInterface         $storeManager,
        ReportLogsFactory             $reportLogsFactory
    )
    {
        parent::__construct($context);
        $this->logger = $logger;
        $this->categoryCollectionFactory = $categoryCollectionFactory;
        $this->conditionChain = $conditionChain;
        $this->request = $request;
        $this->storeManager = $storeManager;
        $this->reportLogsRepository = $reportLogsRepository;
        $this->reportLogsFactory = $reportLogsFactory;
    }

    /**
     * @inheritDoc
     */
    public function execute()
    {
        try {
            $force_param = $this->request->getParam('force');
            $store_id = $this->getRequest()->getParam('store', Store::DEFAULT_STORE_ID);

            $this->logger->info("Translate all categories: Force param = $force_param, store_id = $store_id");

            try {
                $store = $this->storeManager->getStore($store_id);
            } catch (NoSuchEntityException $e) {
                $this->reportLogsRepository->save(
                    $this->createReportLog("Cannot find store id with id: $store_id", Store::DEFAULT_STORE_ID)
                );
                return;
            }

            /**
             * Retrieve the categories to be translated.
             * @var Collection $collection
             */
            $categories = $this->categoryCollectionFactory->create()
                ->setStoreId($store->getId())
                ->addAttributeToSelect('url_key')
                ->addAttributeToSelect('exclude_clonable_auto_translation')
                ->addFieldToFilter('entity_id', ['nin' => [1, $store->getRootCategoryId()]]); // remove the root category

            /**
             * Important behavior!
             * ------------------------------------------
             * This sort function is very important and makes sure the categories don't get screwed up.
             *
             * Because of the way the magento rewrites work, we need
             * to make sure that we first translate the parent category
             * in other words... a breadth first approach!
             *
             * Because Magento stores the category hierarchy like an idiot,
             * we can just sort the collection based on the number of slashes
             * in the path variable of a category.
             * More slashes == deeper category
             *
             * @var CategoryInterface[] $category_array
             */
            $category_array = $categories->getItems();
            usort($category_array, function ($a, $b) {
                $count_a = substr_count($a->getPath(), '/');
                $count_b = substr_count($b->getPath(), '/');
                return $count_a <=> $count_b;
            });

            foreach ($category_array as $index => $category) {
                $category->setStoreId($store->getId());
                $this->conditionChain->process($category, ($force_param == 1), $index);
            }

        } catch (\Throwable $exception) {
            $this->logger->error("Execute TranslateAllCategories: " . $exception->getMessage());
        }
    }

    private function createReportLog(string $message, int $store_id): ReportLogs
    {
        $reportLog = $this->reportLogsFactory->create();
        $reportLog->setMessage($message);
        $reportLog->setStoreId($store_id);
        return $reportLog;
    }
}