<?php

namespace Clonable\Translator\Model;

use Clonable\Translator\Exception\InvalidEntityTypeException;
use Magento\Catalog\Api\Data\CategoryInterface;
use Magento\Catalog\Api\Data\ProductInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\App\CacheInterface;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Store\Model\Store;
use Magento\UrlRewrite\Model\UrlPersistInterface;
use Magento\UrlRewrite\Model\UrlFinderInterface;
use Clonable\Translator\Api\Data\TranslatorInterface;
use Clonable\Translator\Api\Service\ClonableTranslatorApiInterface;
use Magento\UrlRewrite\Service\V1\Data\UrlRewrite as UrlDataRewrite;
use Magento\UrlRewrite\Service\V1\Data\UrlRewriteFactory as UrlRewriteDataFactory;
use Magento\UrlRewrite\Model\ResourceModel\UrlRewriteCollectionFactory;


abstract class AbstractTranslator implements TranslatorInterface
{
    protected UrlPersistInterface $urlPersist;
    protected UrlFinderInterface $urlFinder;
    protected CacheInterface $cache;
    protected Json $json;
    protected StoreManagerInterface $storeManager;
    protected ConfigManager $configManager;
    protected ReportLogsFactory $reportLogsFactory;
    protected ScopeConfigInterface $scopeConfig;
    protected UrlRewriteDataFactory $urlRewriteDataFactory;
    protected UrlRewriteCollectionFactory $urlRewriteCollectionFactory;
    private ClonableTranslatorApiInterface $clonableTranslatorAPIInterface;

    /**
     * @param UrlPersistInterface $urlPersist
     * @param UrlFinderInterface $urlFinder
     * @param CacheInterface $cache
     * @param Json $json
     * @param StoreManagerInterface $storeManager
     * @param ConfigManager $configManager
     * @param ClonableTranslatorApiInterface $clonableTranslatorAPIInterface
     */
    public function __construct
    (
        UrlPersistInterface $urlPersist,
        UrlFinderInterface $urlFinder,
        CacheInterface $cache,
        Json $json,
        StoreManagerInterface $storeManager,
        ConfigManager $configManager,
        ClonableTranslatorApiInterface $clonableTranslatorAPIInterface,
        ScopeConfigInterface $scopeConfig,
        UrlRewriteDataFactory          $urlRewriteFactory,
        UrlRewriteCollectionFactory    $urlRewriteCollectionFactory,
        ReportLogsFactory              $reportLogsFactory
    ){
        $this->urlPersist = $urlPersist;
        $this->urlFinder = $urlFinder;
        $this->cache = $cache;
        $this->json = $json;
        $this->storeManager = $storeManager;
        $this->configManager = $configManager;
        $this->clonableTranslatorAPIInterface = $clonableTranslatorAPIInterface;
        $this->scopeConfig = $scopeConfig;
        $this->reportLogsFactory = $reportLogsFactory;
        $this->urlRewriteDataFactory = $urlRewriteFactory;
        $this->urlRewriteCollectionFactory = $urlRewriteCollectionFactory;
    }

    /**
     * @param string $text
     * @param Store $store
     * @return string
     */
    public function translateText(string $text, Store $store): string
    {
        return ucfirst($this->clonableTranslatorAPIInterface->translateText($text, $store));
    }

    /**
     * @param string $urlKey
     * @param Store $store
     * @return string
     */
    public function translateUrlKey(string $urlKey, Store $store): string
    {
        return $this->clonableTranslatorAPIInterface->translateUrlKey($urlKey, $store);
    }

    /**
     * The method that is responsible for handling the actual translation of a model.
     * The input accepts a JSON encoded string.
     * The required JSON data is depended on the implementation of the actual interface.
     * This method should raise an exception when the required data is not available.
     *
     * @param string $body the message body of the queue message
     * @return void
     **/
    public abstract function translate(string $body): void;

    /**
     * @return array
     */
    protected function getStoresForTranslate(): array
    {
        $storeIds = [];

        foreach ($this->storeManager->getStores() as $store) {
            if ($this->configManager->getApiKey($store->getId()) && $this->configManager->isEnabled($store->getId())) {
                $storeIds[] = $store->getId();
            }
        }

        return $storeIds;
    }

    /**
     *
     * @param string $message
     * @param string $storeId
     * @return ReportLogs
     */
    protected function createReportLog(string $message, string $storeId = '0'): ReportLogs {
        $reportLog = $this->reportLogsFactory->create();
        $reportLog->setMessage($message);
        $reportLog->setStoreId($storeId);
        return $reportLog;
    }

    /**
     * Generate a UrlDataRewrite for an entity.
     * This makes sure that the original URL gets redirected to the translated one.
     *
     * @param CategoryInterface|ProductInterface $entity either a product or a category
     * @param Store $store
     * @param string $original_path
     * @return array
     * @throws InvalidEntityTypeException
     */
    protected function getRedirectDataObject($entity, Store $store, string $original_path): array {
        if ($entity instanceof ProductInterface) {
            $catalog_string = 'product';
        } else if ($entity instanceof CategoryInterface) {
            $catalog_string = 'category';
        } else {
            throw new InvalidEntityTypeException(__("provided entity type not supported for redirects"));
        }
        $suffix = $this->getSuffix($catalog_string, $store);
        if ($suffix !== '' && substr_compare($original_path, $suffix, -strlen($suffix)) !== 0) {
            $original_path .= $suffix;
        }

        $target_path = $entity->getData( $catalog_string === 'category' ? 'url_path' : 'url_key');
        if (substr_compare($target_path, $suffix, -strlen($suffix)) !== 0) {
            $target_path .= $suffix;
        }

        return [
            UrlDataRewrite::ENTITY_TYPE => $catalog_string,
            UrlDataRewrite::ENTITY_ID => $entity->getId(),
            UrlDataRewrite::STORE_ID => $store->getId(),
            UrlDataRewrite::REQUEST_PATH => $original_path,
            UrlDataRewrite::TARGET_PATH => $target_path,
            UrlDataRewrite::REDIRECT_TYPE => 301,
            UrlDataRewrite::IS_AUTOGENERATED => 0,
        ];
    }

    /**
     * @param ProductInterface|CategoryInterface $entity
     * @param Store $store
     * @param string $original_path
     * @param UrlDataRewrite[] $rewrites
     * @return UrlDataRewrite|null
     * @throws InvalidEntityTypeException
     */
    protected function getRedirect($entity, Store $store, string $original_path, array $rewrites): ?UrlDataRewrite {
        $catalog_string = $entity instanceof ProductInterface ? 'product' : 'category';

        try {
            $redirect_data_object = $this->getRedirectDataObject($entity, $store, $original_path);
        } catch (NoSuchEntityException $e) {
            // don't bother creating one if it's going to fail.
            // redirects are technically not required for the correct translations of the urls keys
            return null;
        }

        // Make sure to check the auto generated rewrites for conflicting paths.
        foreach ($rewrites as $rewrite) {
            if ($rewrite->getRequestPath() === $original_path) {
                return null;
            }
        }

        // Make a collection query to check existing rewrites for this category
        $collection = $this->urlRewriteCollectionFactory->create()
            ->addFieldToFilter('entity_type', $catalog_string)
            ->addFieldToFilter('entity_id', $entity->getId())
            ->addFieldToFilter('store_id', $store->getId())
            ->addFieldToFilter('request_path', $original_path)
            ->addFieldToFilter('redirect_type', 301);

        $existing_rewrite = $collection->getFirstItem();
        if ($existing_rewrite->getId()) {
            return null; // if a rewrite for this exact category already exists, then ignore the creation.
        }
        return $this->urlRewriteDataFactory->create(['data' => $redirect_data_object]);
    }

    protected function getSuffix(string $catalog_string, Store $store): string {
        $suffix = (string) $this->scopeConfig->getValue(
            "catalog/seo/{$catalog_string}_url_suffix",
            ScopeInterface::SCOPE_STORE,
            $store->getId()
        );

        if ($suffix !== '' && $suffix[0] !== '.') {
            // Normalize the suffix so it always starts with a dot (.)
            $suffix = '.' . ltrim($suffix, '.');
        }

        return $suffix;
    }
}
