<?php

namespace Clonable\Translator\Model;

use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Model\ResourceModel\ReportLogs as ReportLogsResource;
use Clonable\Translator\Model\ReportLogsFactory;
use Exception;
use Magento\Framework\Exception\CouldNotSaveException;
use Magento\Framework\Exception\NoSuchEntityException;

class ReportLogsRepository implements ReportLogsRepositoryInterface
{
    /**
     * @var ReportLogsResource
     */
    protected ReportLogsResource $resource;

    /**
     * @var ReportLogsFactory
     */
    protected ReportLogsFactory $reportLogsFactory;

    /**
     * ReportLogsRepository constructor.
     *
     * @param ReportLogsResource $resource
     * @param ReportLogsFactory $reportLogsFactory
     */
    public function __construct(
        ReportLogsResource $resource,
        ReportLogsFactory $reportLogsFactory
    ) {
        $this->resource = $resource;
        $this->reportLogsFactory = $reportLogsFactory;
    }

    /**
     * Save log entry.
     *
     * @param \Clonable\Translator\Api\Data\ReportLogsInterface $log
     * @return \Clonable\Translator\Api\Data\ReportLogsInterface
     * @throws CouldNotSaveException
     */
    public function save(\Clonable\Translator\Api\Data\ReportLogsInterface $log): \Clonable\Translator\Api\Data\ReportLogsInterface
    {
        try {
            $this->resource->save($log);
        } catch (Exception $e) {
            throw new CouldNotSaveException(__('Could not save the log: %1', $e->getMessage()));
        }
        return $log;
    }

    /**
     * Get log entry by ID.
     *
     * @param int $id
     * @return \Clonable\Translator\Api\Data\ReportLogsInterface
     * @throws NoSuchEntityException
     */
    public function getById(int $id): \Clonable\Translator\Api\Data\ReportLogsInterface
    {
        $log = $this->reportLogsFactory->create();
        $this->resource->load($log, $id);
        if (!$log->getId()) {
            throw new NoSuchEntityException(__('Log with ID "%1" does not exist.', $id));
        }
        return $log;
    }

    /**
     * @throws NoSuchEntityException
     * @throws Exception
     * @param string $id
     * @return void
     */
    public function deleteById($id): void
    {
        $log = $this->reportLogsFactory->create();
        $this->resource->load($log, $id);
        if (!$log->getId()) {
            throw new NoSuchEntityException(__('Log with ID "%1" does not exist.', $id));
        }
        $this->resource->delete($log);
    }
}
