<?php

namespace Clonable\Translator\Model\Queue\Consumer;

use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Model\Logger\Logger;
use Clonable\Translator\Model\ReportLogs;
use Magento\Framework\MessageQueue\ConsumerConfigurationInterface;
use Magento\Framework\MessageQueue\ConsumerInterface;
use Magento\Framework\MessageQueue\EnvelopeInterface;
use Magento\Framework\MessageQueue\QueueInterface;
use Clonable\Translator\Model\ReportLogsFactory;
use Magento\Framework\Serialize\Serializer\Json;
use ReflectionClass;
use Throwable;

abstract class BaseConsumer implements ConsumerInterface
{
    private const MAX_RETRIES = 50;

    private ConsumerConfigurationInterface $configuration;
    protected ReportLogsRepositoryInterface $reportLogsRepository;
    private ReportLogsFactory $reportLogsFactory;
    protected Json $json;
    private Logger $logger;

    public function __construct
    (
        ConsumerConfigurationInterface $configuration,
        ReportLogsRepositoryInterface  $reportLogsRepository,
        ReportLogsFactory $reportLogsFactory,
        JSON $json,
        Logger $logger
    ){
        $this->configuration = $configuration;
        $this->reportLogsRepository = $reportLogsRepository;
        $this->reportLogsFactory = $reportLogsFactory;
        $this->json = $json;
        $this->logger = $logger;
    }

    /**
     * @param int|null $maxNumberOfMessages
     */
    public function process($maxNumberOfMessages = null): void
    {
        $class = new ReflectionClass($this);
        $class_name = $class->getShortName();

        $queue = $this->configuration->getQueue();
        $max_number_of_messages = $maxNumberOfMessages ?? 100;
        $processed = 0;

        try {
            for ($i = 0; $i < $max_number_of_messages; $i++) {
                $message = $queue->dequeue();

                if (null !== $message) {
                    $retries = $message->getProperties()['retries'] ?? null;
                    $this->logger->debug("Dequeued message:" . PHP_EOL . print_r($message->getProperties(), true));

                    // Check if the maximum number of retries is reached.
                    // If for some reason, the number of retries is null, we'll assume that we can requeue the message.
                    if ($retries == null || $retries < self::MAX_RETRIES) {
                        $this->handleMessage($queue, $message);
                        $processed++;
                    } else {
                        $max_retries = self::MAX_RETRIES;
                        $this->reportLogsRepository->save($this->createReportLog("[ERROR] Maximum number of retries ($max_retries) reached."));
                        $queue->reject($message, false,"Maximum number of retries reached.");
                    }
                } else {
                    // noop for an empty message, happens when the total items on the queue are less that max_number_of_messages
                    break;
                }
            }

            if ($processed > 0) {
                $processed_messages = number_format($processed);
                $this->reportLogsRepository->save($this->createReportLog("[GOOD] Successfully processed $class_name translations: total items = $processed_messages"));
            }
        } catch (Throwable $exception) {
            $this->reportLogsRepository->save($this->createReportLog("$class_name consumer error: {$exception->getMessage()}"));
        }
    }

    public abstract function handleMessage(QueueInterface $queue, EnvelopeInterface $message);

    /**
     *
     * @param string $message
     * @param string $storeId
     * @return ReportLogs
     */
    protected function createReportLog(string $message, string $storeId = '0'): ReportLogs {
        $reportLog = $this->reportLogsFactory->create();
        $reportLog->setMessage($message);
        $reportLog->setStoreId($storeId);
        return $reportLog;
    }
}