<?php

namespace Clonable\Translator\Model\Queue\Consumer;

use Clonable\Translator\Api\ReportLogsRepositoryInterface;
use Clonable\Translator\Exception\CategoryTranslationException;
use Clonable\Translator\Model\Category\CategoryTranslator;
use Clonable\Translator\Model\Logger\Logger;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\MessageQueue\ConsumerConfigurationInterface;
use Magento\Framework\MessageQueue\EnvelopeInterface;
use Magento\Framework\MessageQueue\QueueInterface;
use Clonable\Translator\Model\ReportLogsFactory;
use Magento\Framework\Serialize\Serializer\Json;
use Magento\Framework\App\ResourceConnection;
use Magento\Catalog\Model\CategoryFactory;
use Magento\Store\Api\Data\StoreInterface;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;

class Category extends BaseConsumer
{
    private CategoryTranslator $translatorCategory;
    private Logger $logger;
    private ResourceConnection $resourceConnection;
    private CategoryFactory $categoryFactory;
    private StoreManagerInterface $storeManager;

    public function __construct
    (
        ConsumerConfigurationInterface $configuration,
        ReportLogsRepositoryInterface  $reportLogsRepository,
        ReportLogsFactory              $reportLogsFactory,
        CategoryTranslator             $translatorCategory,
        Json                           $json,
        Logger                         $logger,
        ResourceConnection             $resourceConnection,
        CategoryFactory                $categoryFactory,
        StoreManagerInterface          $storeManager
    )
    {
        parent::__construct($configuration, $reportLogsRepository, $reportLogsFactory, $json, $logger);
        $this->translatorCategory = $translatorCategory;
        $this->logger = $logger;
        $this->resourceConnection = $resourceConnection;
        $this->categoryFactory = $categoryFactory;
        $this->storeManager = $storeManager;
    }

    public function handleMessage(QueueInterface $queue, EnvelopeInterface $message)
    {
        try {
            $decoded = $this->json->unserialize($message->getBody());

            $data = $this->json->unserialize($decoded);
            $store_id = intval($data['storeId'] ?? 0);
            $store = $this->getStore($store_id);

            if ($store === null) {
                $this->reportLogsRepository->save($this->createReportLog("[ERROR] the correct store could not be retrieved, cancelling translation."));
                $queue->reject($message, false, "no store specified");
                return;
            }

            $category_id = intval($data['categoryId'] ?? 0);
            $debugging_index = $data['debugging_index'] ?? null;

            $parent_id = $this->getParentId($category_id);
            if ($parent_id === Store::DEFAULT_STORE_ID || $parent_id == $store->getRootCategoryId()) {
                // These parents are and should never be translated.
                // Therefore, we do not have to check the parent attribute.
                $this->logger->debug("[DEBUG:$debugging_index] The parent is the root category, we can translate this immediately.");
                $this->translatorCategory->translate($decoded);
                $queue->acknowledge($message);
                return;
            }

            /**
             * Due to how Magento works, it's important to check whether the parent is translated or not.
             * If for some reason, the parent is not translated, we should requeue the child.
             *
             * This happens when the dequeue-ing logic of Magento screws up and does not conform to FIFO.
             */
            $parent_exclude_clonable_auto_translation = $this->getCategoryAttributeValue($parent_id, 'exclude_clonable_auto_translation', $store_id);
            if ($parent_exclude_clonable_auto_translation == 0) {
                // parent is definitely not translated, show this item should requeue
                $this->logger->debug("[DEBUG:$debugging_index] Parent ($parent_id) was not translated yet, requeue this item (original queue position: $debugging_index)");
                $queue->reject($message, true, "Parent was not translated yet, requeue this item");
                return;
            }

            $this->logger->debug("Category processing index: $debugging_index");
            $this->translatorCategory->translate($decoded);
            $queue->acknowledge($message);
        } catch (CategoryTranslationException $exception) {
            $this->reportLogsRepository->save($this->createReportLog("[ERROR] in Category consumer: " . $exception->getMessage()));
            $queue->reject($message, $exception->shouldRequeue(), $exception->getMessage());
        } catch (\Throwable $exception) {
            $this->reportLogsRepository->save($this->createReportLog("[CRASH] in Category consumer: " . $exception->getMessage()));
            $this->logger->critical($exception->getMessage());
            $queue->reject($message, true, $exception->getMessage());
        }
    }

    public function getStore(int $store_id): ?StoreInterface{
        try {
            return $this->storeManager->getStore($store_id);
        } catch (NoSuchEntityException $e) {
            $this->reportLogsRepository->save(
                $this->createReportLog("Cannot find store id with id: $store_id", Store::DEFAULT_STORE_ID)
            );
            return null;
        }
    }

    /**
     * Retrieve the ID of the parent of a given category
     * @param int $category_id
     * @return int
     */
    private function getParentId(int $category_id): int {
        $connection = $this->resourceConnection->getConnection();
        $table = $this->resourceConnection->getTableName('catalog_category_entity');

        $parent_id = $connection->fetchOne(
            "SELECT parent_id FROM $table WHERE entity_id = ?",
            [$category_id]
        );
        return (int) $parent_id;
    }

    /**
     * @param int $category_id
     * @param string $attribute
     * @param int $store_id
     * @return int|null
     */
    private function getCategoryAttributeValue(int $category_id, string $attribute, int $store_id): ?int {
        $connection = $this->resourceConnection->getConnection();
        $eav_table_name = $this->resourceConnection->getTableName('eav_attribute');
        $category_int_table = $this->resourceConnection->getTableName('catalog_category_entity_int');

        $attribute_id = $connection->fetchOne(
            "SELECT attribute_id FROM $eav_table_name WHERE attribute_code = ? AND entity_type_id = 3",
            [$attribute]
        );

        $value = $connection->fetchOne(
            "SELECT value FROM $category_int_table WHERE attribute_id = ? AND entity_id = ? AND store_id = ?",
            [$attribute_id, $category_id, $store_id]
        );

        if (empty($value)) {
            $value = $connection->fetchOne(
                "SELECT value FROM $category_int_table WHERE attribute_id = ? AND entity_id = ? AND store_id = 0",
                [$attribute_id, $category_id]
            );
        }

        return empty($value) ? null : (int) $value;
    }
}
