<?php

namespace Clonable\Translator\Model\Service;

use Clonable\Translator\Exception\TranslationUnavailableException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\HTTP\Client\Curl;
use Magento\Store\Model\Store;
use Clonable\Translator\Api\Service\ClonableTranslatorApiInterface;
use Clonable\Translator\Model\ConfigManager;
use Clonable\Translator\Model\Logger\Logger;

class ClonableTranslatorApi implements ClonableTranslatorApiInterface
{
    /**
     * @var Curl
     */
    private Curl $curl;

    /**
     * @param ConfigManager $configManager
     */
    private ConfigManager $configManager;

    private Logger $logger;

    public function __construct(
        Curl $curl,
        ConfigManager $configManager,
        Logger $logger
    ){
        $this->curl = $curl;
        $this->configManager = $configManager;
        $this->logger = $logger;
    }

    /**
     * Translate a URL using the external API.
     *
     * @param string $urlKey
     * @param Store $store
     * @return string
     * @throws LocalizedException
     */
    public function translateUrlKey(string $urlKey, Store $store): string
    {
        $maxAttempts = 5;
        $initialDelay = 1;
        $data = $this->prepareUrlTranslationData($urlKey, $store);
        $headers = $this->prepareHeaders($store);

        $this->logger->info('Request Body: ' . json_encode($data));

        for ($attempt = 0; $attempt < $maxAttempts; $attempt++) {
            $this->curl->setHeaders($headers);
            $this->curl->post(ConfigManager::API_URL_TRANSLATE, json_encode($data));

            if ($this->curl->getStatus() === 200) {
                $response = $this->curl->getBody();
                return basename(parse_url(json_decode($response)->translation, PHP_URL_PATH));
            } else {
                sleep($initialDelay * pow(2, $attempt));
            }
        }

        throw new LocalizedException(__('Failed to translate URL after multiple attempts.'));
    }

    /**
     * Translate a product name using the external API.
     *
     * @param string $text
     * @param Store $store
     * @return string the translated text
     * @throws LocalizedException will be thrown if a specific translation cannot be made.
     * @throws TranslationUnavailableException gets thrown when the no translations will be created.
     */
    public function translateText(string $text, Store $store): string
    {
        $post_data = $this->prepareTextTranslationData($text, $store);
        $headers = $this->prepareHeaders($store);

        $error_description = "internal server error";
        $retry = false;
        $attempt_number = 0;
        // max retries of 12 should always be enough, the current max limit is 60 requests a minit
        // when taking into account that sleep takes 5 seconds, 12 should fit perferctly.
        $max_attempts = 12;

        do {
            $this->curl->setHeaders($headers);
            $this->curl->post(ConfigManager::API_URL_TEXT, json_encode($post_data));

            $status_code = $this->curl->getStatus();
            $response = $this->curl->getBody();
            $json = json_decode($response);

            if ($status_code === 200) {
                if ($json->translation_done) {
                    return $json->translation;
                }
                throw new TranslationUnavailableException(__("Translation API is currently unavailable."));
            } else if ($status_code === 429 || $status_code === 503) {
                // Hit the rate limit, just retry it until the translations are available again.
                $retry = true;
                if ($attempt_number >= $max_attempts) {
                    // if for some reason this gets this high, then just stop the translations.
                    $error_description = "Too many attempts. Please try again later.";
                    break;
                }
                $attempt_number++;
                sleep(5);
            } else if ($status_code >= 400 && $status_code < 500) {
                if ($status_code === 401 || $status_code === 403) {
                    $error_description = "Unauthorized access. Make sure you're using the API key from your account.";
                } else {
                    // client error, probably some structural issues with the request, retry logic is not needed
                    $error_description = $json->error_description ?? 'unknown error';
                }
            }
        } while ($retry);

        throw new LocalizedException(__("Translation of text failed with status code $status_code: $error_description"));
    }

    /**
     * Prepare data for URL translation.
     *
     * @param string $urlKey
     * @param Store $store
     * @return array
     */
    private function prepareUrlTranslationData(string $urlKey, Store $store): array
    {
        return [
            'url' => $this->prepareUrl($urlKey, $store),
            'clone_id' => $this->configManager->getCloneId($store->getStoreId()),
            'site_id' => $this->configManager->getSiteId($store->getStoreId()),
            'original_language' => $this->configManager->getOriginalLanguage($store->getStoreId()),
            'target_language' => $this->configManager->getTargetLanguage($store->getStoreId()),
            'create_new' => true
        ];
    }

    /**
     * Make sure the url format is build up correctly based on the settings of the clone and the store.
     * @param string $urlKey the key that needs to be translated
     * @param Store $store the store to create the translation for
     * @return string the full url that needs to be translated
     */
    private function prepareUrl(string $urlKey, Store $store): string {
        $baseUrl = $store->getBaseUrl(); // base url always ends with a slash
        $cloneSubfolder = $this->configManager->getSubfolderClone($store);

        if ($cloneSubfolder === null) {
            return $baseUrl . $urlKey; // do not add subfolder, clone does not have it.
        } elseif ($this->strEndsWith($baseUrl, $cloneSubfolder)) {
            return $baseUrl . $urlKey; // do not add subfolder, baseUrl already contains it.
        } else {
            // make sure to add the subfolder after the base url, and make sure not to add a double slash
            $baseUrlWithoutSlash = substr($baseUrl, 0, -1);
            return $baseUrlWithoutSlash . $cloneSubfolder . $urlKey;
        }
    }

    private function strEndsWith(string $haystack, string $needle): bool {
        if ($haystack === '' && $needle !== '') {
            return false;
        }
        $len = strlen($needle);
        return substr_compare($haystack, $needle, -$len, $len) === 0;
    }

    /**
     * Prepare data for product name translation.
     *
     * @param string $to_translate
     * @param Store $store
     * @return array
     */
    private function prepareTextTranslationData(string $to_translate, Store $store): array
    {
        return [
            "content" => $to_translate,
            "text_only" => true,
            "editor_mode" => false,
            "clone_id" => $this->configManager->getCloneId($store->getStoreId()),
        ];
    }

    /**
     * Prepare headers for API requests.
     *
     * @param Store $store
     * @return array
     */
    private function prepareHeaders(Store $store): array
    {
        $api_key = $this->configManager->decryptApiKey($this->configManager->getApiKey($store->getStoreId()));
        return [
            'Authorization' => "Bearer $api_key",
            'Accept' => 'application/json',
            'Content-Type' => 'application/json'
        ];
    }
}
